/* Font manager.

   Copyright (C) 2000 Daiki Ueno <ueno@unixuser.org>

   Author: Daiki Ueno <ueno@unixuser.org>
   Created: 2000-05-16

   This file is part of UltraPoint.

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2, or (at your option)  
   any later version.                                                   

   This program is distributed in the hope that it will be useful,      
   but WITHOUT ANY WARRANTY; without even the implied warranty of       
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the        
   GNU General Public License for more details.                         

   You should have received a copy of the GNU General Public License    
   along with GNU Emacs; see the file COPYING.  If not, write to the    
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,         
   Boston, MA 02111-1307, USA.                                          

*/

#include <string.h>
#include "font.h"

static long convert_8bit (UptCharset *charset, const unicode_char_t codepoint);
static long convert_euc (UptCharset *charset, const unicode_char_t codepoint);

static guint find_char_mask (unicode_char_t ucs);

typedef struct _CharRange CharRange;
struct _CharRange
{
  guint16 start;
  guint16 end;
  guint16 charsets;
};

#include "charset-range.tbl"

static UptCharset charset_cache[] = {
  { "ISO-8859-1", "iso8859_1", convert_8bit, FALSE },
  { "ISO-8859-2", "iso8859_2", convert_8bit, FALSE },
  { "ISO-8859-3", "iso8859_3", convert_8bit, FALSE },
  { "ISO-8859-4", "iso8859_4", convert_8bit, FALSE },
  { "ISO-8859-5", "iso8859_5", convert_8bit, FALSE },
  { "ISO-8859-6", "iso8859_6", convert_8bit, FALSE },
  { "ISO-8859-7", "iso8859_7", convert_8bit, FALSE },
  { "ISO-8859-8", "iso8859_8", convert_8bit, FALSE },
  { "ISO-8859-9", "iso8859_9", convert_8bit, FALSE },
  { "EUC-JP", "jisx0208", convert_euc, TRUE },
  { "EUC-KR", "gb2312", convert_euc, TRUE },
  { "EUC-KR", "ksc5601", convert_euc, TRUE }
};

static const char *family_map[] = { "f", "t", "h", "c" };
static const char *face_map[] = { "", "B", "I", "BI" };

static GSList *font_cache = NULL;

static guint
find_char_mask (ucs)
     unicode_char_t ucs;
{
  int start, end, middle;

  start = 0;
  end = G_N_ELEMENTS (ranges) - 1;

  if (ranges[start].start > ucs || ranges[end].end < ucs)
    return 0;

  while (1)
    {
      middle = (start + end) / 2;
      if (middle == start)
        {
          if (ranges[middle].start > ucs || ranges[middle].end < ucs)
            return CHARSET_ISO_10646;
          else
            return ranges[middle].charsets | CHARSET_ISO_10646;
        }
      else
        {
          if (ranges[middle].start <= ucs)
            start = middle;
          else if (ranges[middle].end >= ucs)
            end = middle;
          else
            return CHARSET_ISO_10646;
        }
    }
}

static long
convert_8bit (charset, codepoint)
     UptCharset *charset;
     const unicode_char_t codepoint;
{
  unsigned char inbuf[16];
  char outbuf;
  const char *inptr = inbuf;
  int inbytesleft = 4;
  char *outptr = &outbuf;
  int outbytesleft = 1;
  int i;

  for  (i = 0; i < inbytesleft; i++)
    inbuf[i] = (codepoint >> ((inbytesleft - i - 1) * 8) & 0xff);

  unicode_iconv (charset->iconv_cd,
		 (const char **)&inptr, &inbytesleft,
		 &outptr, &outbytesleft);

  return (long)(outbuf&0xff);
}

static long
convert_euc (charset, codepoint)
     UptCharset *charset;
     const unicode_char_t codepoint;
{
  unsigned char inbuf[16];
  char outbuf[16];
  const char *inptr = inbuf;
  int inbytesleft = 4;
  char *outptr = outbuf;
  int outbytesleft = 16;
  int i;

  for  (i = 0; i < inbytesleft; i++)
    inbuf[i] = (codepoint >> ((inbytesleft - i - 1) * 8) & 0xff);

  unicode_iconv (charset->iconv_cd,
		 (const char **)&inptr, &inbytesleft,
		 &outptr, &outbytesleft);

  if ((guchar)outbuf[0] < 0x80)
    return outbuf[0];
  else
    return ((guchar)outbuf[0] & 0x7f) << 8 | ((guchar)outbuf[1] & 0x7f);
}

static void
font_free (font)
     UptFont *font;
{
  if (font->id > -1)
    VF_CloseFont (font->id);
  font_cache = g_slist_remove (font_cache, font);
  if (font->charset)
    upt_object_unref (font->charset);
  g_free (font);
}

UptFont *
upt_font_new (pixel_size, family_id, face_id)
     gint pixel_size, family_id, face_id;
{
  UptFont *font;

  font = g_new0 (UptFont, 1);
  g_assert (font != NULL);

  font->pixel_size = pixel_size;
  font->family_id = family_id;
  font->face_id = face_id;
  font->id = -1;
  font->free = font_free;

  return font;
}

static UptFont *
find_font_from_cache (mask, font_pixel_size, font_family, font_face)
     guint mask;
     gint font_pixel_size, font_family, font_face;
{
  GSList *p;
  UptFont *font;

  for (p = font_cache; p; p = g_slist_next (p))
    {
      font = p->data;
      if ((mask & (1 << font->charset_id))
	  && font->pixel_size == font_pixel_size
	  && font->family_id == font_family
	  && font->face_id == font_face)
	return font;
    }

  return NULL;
}

static UptFont *
find_font_1 (charset_id, font_pixel_size, font_family, font_face)
     gint charset_id;
     gint font_pixel_size, font_family, font_face;
{
  gint font_id;
  gchar vflib_name[32];
  UptCharset *charset;
  UptFont *font;

  charset = &charset_cache[charset_id];

  vflib_name[0] = '\0';
  g_snprintf (vflib_name, sizeof(vflib_name), "%s:%d%s:%s",
	      charset->vflib_name, font_pixel_size,
	      face_map[font_face], family_map[font_family]);
  font_id =
    VF_OpenFont2 (vflib_name, font_pixel_size, 1, 1);

  if (font_id == -1)
    {
      vflib_name[0] = '\0';
      g_snprintf (vflib_name, sizeof(vflib_name), "%s:n%s:%s",
		  charset->vflib_name, face_map[font_face],
		  family_map[font_family]);
      font_id = VF_OpenFont2 (vflib_name, font_pixel_size, 1, 1);
    }

  if (font_id == -1)
    return NULL;

  font = upt_font_new (font_pixel_size, font_family, font_face);
  font->charset_id = charset_id;
  font->id = font_id;
  font->name = vflib_name;
  font->charset = charset;

  if (strncmp (charset->iconv_encoding, "UNICODE", 7)
      && !charset->iconv_cd)
    charset->iconv_cd =
      unicode_iconv_open (charset->iconv_encoding, "UCS4");
  g_assert (charset->iconv_cd != (unicode_iconv_t)-1);
  font_cache = g_slist_append (font_cache, font);

  return font;
}

UptFont *
upt_find_font (ucs, font_pixel_size, font_family, font_face)
     unicode_char_t ucs;
     gint font_pixel_size, font_family, font_face;
{
  guint mask = find_char_mask (ucs);
  gint i;
  UptFont *font;
 
  font = find_font_from_cache (mask, font_pixel_size, font_family, font_face);
  for (i = 0; !font && i < (sizeof(mask)<<3); i++)
    {
      if (!(mask & (1 << i)))
	continue;
      font = find_font_1 (i, font_pixel_size, font_family, font_face);
    }
  if (!font)
    g_warning (_("Cannot find font for UCS (%x)."), ucs);

  return font;
}

void
upt_clear_font_cache (void)
{
  GSList *p;

  for (p = font_cache; p; p = g_slist_next (p))
    {
      UptFont *font = p->data;
      if (font->id > 0)
	VF_CloseFont (font->id);
      g_free (font);
    }
  g_slist_free (font_cache);
  font_cache = NULL;
}
