/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2006)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2006)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "gtk/gtkwidget.h"

#include <GL/gl.h>

#include "gtk_openGLWidget.h"
#include "OSOpenGL/visu_openGL.h"
#include "visu_tools.h"

/* OpenGL implementation dependencies */
#ifdef HAVE_GTKGLEXT
  #include <gtk/gtkgl.h>
  #include <gdk/gdkx.h>
  #define IMPL_GTKGLEXT
#else
  #if SYSTEM_X11 == 1
  #include <GL/glx.h>
  #include <gdk/gdkx.h>
  #define IMPL_BUILTIN_X11
  #endif
  #if SYSTEM_WIN32 == 1
  #include <windows.h>
  #include <gdk/gdkwin32.h>
  #define IMPL_BUILTIN_WIN32
  #endif
#endif

/**
 * SECTION:openglwidget
 * @short_description: defines an OpenGL capable GtkWidget.
 *
 * <para>This is a simple implementation of GtkGlExt to create an
 * OpenGL surface that is a full GtkWidget. When creating such a
 * widget, one should give then a RedrawMethod() to tell the widget
 * how to redraw itself when needed.</para>
 *
 * <para>The current implementation is working on X11 (built-in or
 * with GtkGlExt) and Win32.</para>
 */
struct _OpenGLWidget
{
  GtkWidget parent;

  gboolean sizeAllocation_has_run;
  gboolean dispose_has_run;

  /* Redraw method and user data. */
  RedrawMethod redraw;
  gpointer redrawData;

  /* OpenGL part, OS dependent. */
  gboolean isContextDirect;
#ifdef IMPL_GTKGLEXT
  GdkGLConfig *glconfig;
  GdkGLWindow *glwindow;
  GdkGLContext *context;
#endif
#ifdef IMPL_BUILTIN_X11
  Display *dpy;
  XVisualInfo *vinfo;
  GLXContext context;
#endif
#ifdef IMPL_BUILTIN_WIN32
  HDC hdc;
  HGLRC context;
  HWND windowId;
#endif
};

struct _OpenGLWidgetClass
{
  GtkWidgetClass parent_class;

  OpenGLWidget *contextCurrent;
};

G_DEFINE_TYPE(OpenGLWidget, openGLWidget, GTK_TYPE_WIDGET)

/* Local variables. */
static OpenGLWidgetClass *myClass = (OpenGLWidgetClass*)0;

/* Local callbacks. */
static gboolean openGLWidgetEvent_expose(GtkWidget *widget, GdkEventExpose *event);
static gboolean openGLWidgetEvent_visibility(GtkWidget *widget, GdkEventVisibility *event);
static void openGLWidgetEvent_sizeRequest(GtkWidget *widget, GtkRequisition *requisition);
static void openGLWidgetEvent_sizeAllocate(GtkWidget *widget, GtkAllocation *allocation);
static void openGLWidgetEvent_realise(GtkWidget *widget);

/* Initialisation methods. */
static void openGLWidgetInit_context(OpenGLWidget *render, gboolean contextIsDirect);

/* Freeing methods. */
static void openGLWidgetEvent_dispose(GObject *obj);
static void openGLWidgetEvent_finalize(GObject *obj);
static void openGLWidgetFree_openGL(OpenGLWidget *render);

/* Miscellaneous methods. */
static void openGLWidgetSet_viewport(OpenGLWidget *render, guint width,
				     guint height, gboolean redraw);
static GdkColormap* openGLWidgetGet_openGLColormap(OpenGLWidget *render);
static void openGLWidgetSet_pixelFormat(OpenGLWidget *render);

static void openGLWidget_class_init(OpenGLWidgetClass *class)
{
  GtkWidgetClass *widget_class;
  GObjectClass *gobject_class;

  /* Dealing with GObject events. */
  gobject_class = G_OBJECT_CLASS(class);
  gobject_class->dispose  = openGLWidgetEvent_dispose;
  gobject_class->finalize = openGLWidgetEvent_finalize;

  /* Dealing with widget events. */
  widget_class = GTK_WIDGET_CLASS(class);
  widget_class->expose_event  = openGLWidgetEvent_expose;
  widget_class->visibility_notify_event  = openGLWidgetEvent_visibility;
  widget_class->realize       = openGLWidgetEvent_realise;
  widget_class->size_request  = openGLWidgetEvent_sizeRequest;
  widget_class->size_allocate = openGLWidgetEvent_sizeAllocate;

  class->contextCurrent = (OpenGLWidget*)0;

  myClass = class;
}

static void openGLWidget_init(OpenGLWidget *render)
{
  DBG_fprintf(stderr, "Gtk OpenGL (init) : create object %p.\n", (gpointer)render);
  render->sizeAllocation_has_run = FALSE;
  render->dispose_has_run = FALSE;

  render->redraw = (RedrawMethod)0;
  render->redrawData = (gpointer)0;

  render->isContextDirect = FALSE;
#ifdef IMPL_GTKGLEXT
  render->glconfig = (GdkGLConfig*)0;
  render->glwindow = (GdkGLWindow*)0;
  render->context  = (GdkGLContext*)0;
#endif
#ifdef IMPL_BUILTIN_X11
  render->dpy      = (Display*)0;
  render->vinfo    = (XVisualInfo*)0;
  render->context  = (GLXContext)0;
#endif
#ifdef IMPL_BUILTIN_WIN32
  render->hdc      = (HDC)0;
  render->context  = (HGLRC)0;
  render->windowId = (HWND)0;
#endif

  /* Cancel the GTK double buffering since it is taken into
     account by OpenGL itself. */
  gtk_widget_set_double_buffered(GTK_WIDGET(render), FALSE);
}

GtkWidget* openGLWidgetNew(gboolean contextIsDirect)
{
  OpenGLWidget *render;

  render = OPENGL_WIDGET(g_object_new(TYPE_OPENGL_WIDGET, NULL));

  render->isContextDirect = contextIsDirect;

  return GTK_WIDGET(render);
}

/* Methods to deals with the events. */
static void openGLWidgetEvent_dispose(GObject *obj)
{
  DBG_fprintf(stderr, "Gtk OpenGL (events) : dispose event for object %p.\n", (gpointer)obj);

  if (OPENGL_WIDGET(obj)->dispose_has_run)
    return;

  OPENGL_WIDGET(obj)->dispose_has_run = TRUE;
  /* Chain up to the parent class */
  G_OBJECT_CLASS(openGLWidget_parent_class)->dispose(obj);
}
static void openGLWidgetEvent_finalize(GObject *obj)
{
  DBG_fprintf(stderr, "Gtk OpenGL (events) : finalize event for object %p.\n", (gpointer)obj);

  openGLWidgetFree_openGL(OPENGL_WIDGET(obj));

  /* Chain up to the parent class */
  G_OBJECT_CLASS(openGLWidget_parent_class)->finalize(obj);
}
static gboolean openGLWidgetEvent_expose(GtkWidget *widget, GdkEventExpose *event)
{
  OpenGLWidget *render;

  DBG_fprintf(stderr, "Gtk OpenGL (events): expose event for object %p.\n", (gpointer)widget);

  render = OPENGL_WIDGET(widget);
  if (render->redraw)
    openGLWidgetRedraw(render);
  else
    {
      DBG_fprintf(stderr, " | clear window.\n");
      gdk_window_clear_area(widget->window, event->area.x, event->area.y,
			    event->area.width, event->area.height);
    }

  return FALSE;
}
static gboolean openGLWidgetEvent_visibility(GtkWidget *widget, GdkEventVisibility *event)
{
  OpenGLWidget *render;

  if (event->state != GDK_VISIBILITY_UNOBSCURED)
    return FALSE;

  DBG_fprintf(stderr, "Gtk OpenGL (events): visibility event for object %p.\n", (gpointer)widget);
  render = OPENGL_WIDGET(widget);
  if (render->redraw)
    openGLWidgetRedraw(render);
  else
    {
      DBG_fprintf(stderr, " | clear window.\n");
      gdk_window_clear(widget->window);
    }

  return FALSE;
}
static void openGLWidgetEvent_sizeRequest(GtkWidget *widget, GtkRequisition *requisition)
{
  OpenGLWidget *render;
  int width, height;

  DBG_fprintf(stderr, "Gtk OpenGL (events) : size request event (%dx%d).\n", requisition->width, requisition->height);

  render = OPENGL_WIDGET(widget);
  if (render->sizeAllocation_has_run)
    {
      width = widget->allocation.width;
      height = widget->allocation.height;
    }
  else
    {
      width = 200;
      height = 200;
    }
  widget->requisition.width = width;
  widget->requisition.height = height;
  /* Chain up to default that simply reads current requisition */
  GTK_WIDGET_CLASS(openGLWidget_parent_class)->size_request(widget, requisition);
}
static void openGLWidgetEvent_sizeAllocate(GtkWidget *widget, GtkAllocation *allocation)
{
  OpenGLWidget *render;

  render = OPENGL_WIDGET(widget);
  if ((widget->allocation.width == allocation->width) &&
      (widget->allocation.height == allocation->height))
    return;

  DBG_fprintf(stderr, "Gtk OpenGL (events) : size allocation event (%dx%d).\n", allocation->width, allocation->height);

  render->sizeAllocation_has_run = TRUE;
  /* Chain up to default that simply reads current requisition */
  GTK_WIDGET_CLASS(openGLWidget_parent_class)->size_allocate(widget, allocation);

#ifdef IMPL_BUILTIN_X11
  glXWaitX();
#endif

  openGLWidgetSet_viewport(render, widget->allocation.width, widget->allocation.height, FALSE);
}
static void openGLWidgetEvent_realise(GtkWidget *widget)
{
  OpenGLWidget *render;
  GdkWindowAttr attributes;
  gint attributes_mask;
  GdkColormap *colormap;

  DBG_fprintf(stderr, "Gtk OpenGL (events): realise event for %p.\n",
	      (gpointer)widget);

  render = OPENGL_WIDGET(widget);

#ifndef IMPL_BUILTIN_WIN32
  openGLWidgetSet_pixelFormat(render);
#endif

  colormap = openGLWidgetGet_openGLColormap(render);
  DBG_fprintf(stderr, "Gtk OpenGL (debug): colormap is %p.\n",
	      (gpointer)colormap);

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gdk_colormap_get_visual(colormap);
  attributes.colormap = colormap;
  attributes.event_mask = GDK_EXPOSURE_MASK |
    GDK_BUTTON_PRESS_MASK |
    GDK_BUTTON_RELEASE_MASK |
    GDK_SCROLL_MASK |
    GDK_KEY_PRESS_MASK |
    GDK_POINTER_MOTION_MASK |
    GDK_POINTER_MOTION_HINT_MASK;

#ifdef IMPL_GTKGLEXT
  attributes_mask = GDK_WA_X | GDK_WA_Y;
#else
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
#endif

  widget->window = gdk_window_new(gtk_widget_get_parent_window(widget), 
				  &attributes, attributes_mask);
  gdk_window_set_user_data(widget->window, render);
  DBG_fprintf(stderr, "Gtk OpenGL (debug): GdkWindow is %p.\n",
	      (gpointer)widget->window);

  widget->style = gtk_style_attach(widget->style, widget->window);
  gtk_style_set_background(widget->style, widget->window, GTK_STATE_NORMAL);

  GTK_WIDGET_SET_FLAGS(widget, GTK_CAN_FOCUS);
  GTK_WIDGET_SET_FLAGS(widget, GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);

  gdk_display_sync(gtk_widget_get_display(widget));
#ifdef IMPL_BUILTIN_X11
  glXWaitX();
#endif

  /* Initialize context for OpenGL, OS dependent. */
#ifdef IMPL_BUILTIN_WIN32
  openGLWidgetSet_pixelFormat(render);
#endif
  openGLWidgetInit_context(render, render->isContextDirect);
  openGLWidgetSet_current(render, FALSE);
}


OpenGLWidget* openGLWidgetClassGet_currentContext()
{
  g_return_val_if_fail(myClass, (OpenGLWidget*)0);

  return myClass->contextCurrent;
}

static void openGLWidgetSet_viewport(OpenGLWidget *render, guint width,
				     guint height, gboolean redraw)
{
  g_return_if_fail(IS_OPENGL_WIDGET(render));

  if (OPENGL_WIDGET_GET_CLASS(render)->contextCurrent != render)
    return;

  DBG_fprintf(stderr, " | adjusting viewport to %dx%d.\n", width, height);
  glViewport(0, 0, width, height);
  if (redraw)
    {
      /* We synchronize the rendering area. */
      gdk_display_sync(gtk_widget_get_display(GTK_WIDGET(render)));
      /* We clear the back buffer and swap because this buffer has
	 still the wrong size. */
      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
      openGLWidgetSwap_buffers(render);
    }
}
void openGLWidgetSet_redraw(OpenGLWidget *render, RedrawMethod method, gpointer data)
{
  g_return_if_fail(IS_OPENGL_WIDGET(render));
  
  DBG_fprintf(stderr, "Gtk OpenGL (action) : set redraw method for OpenGL area %p.\n", (gpointer)render);
  render->redraw = method;
  render->redrawData = data;
}
void openGLWidgetRedraw(OpenGLWidget *render)
{
  g_return_if_fail(IS_OPENGL_WIDGET(render));

  if (!render->redraw)
    return;

  DBG_fprintf(stderr, "Gtk OpenGL (action) : redraw OpenGL area for %p.\n", (gpointer)render);
  if (render->redraw)
    {
      DBG_fprintf(stderr, " | set current.\n");
      openGLWidgetSet_current(render, FALSE);
      DBG_fprintf(stderr, " | redraw inside.\n");
      render->redraw((const char**)0, render->redrawData);
      DBG_fprintf(stderr, " | swap buffers.\n");
      openGLWidgetSwap_buffers(render);
    }
}

guchar* openGLWidgetGet_pixmapData(OpenGLWidget *render, int *width,
				   int *height, gboolean offScreen, gboolean hasAlpha)
{
  GtkWidget *wd;
  guchar *image;
  DumpImage *dumpData;

  g_return_val_if_fail(IS_OPENGL_WIDGET(render), (guchar*)0);
  g_return_val_if_fail(OPENGL_WIDGET_GET_CLASS(render)->contextCurrent == render, (guchar*)0);
  g_return_val_if_fail(render->redraw, (guchar*)0);
  g_return_val_if_fail(width && height, (guchar*)0);

  wd = GTK_WIDGET(render);

  if (!offScreen)
    {
      *width = wd->allocation.width;
      *height = wd->allocation.height;
      return visuOpenGLGet_pixmapData(*width, *height, hasAlpha);
    }

  /* If a method is given, then we draw
     in memory to a pixmap. */
  *width = (*width > 0)?*width:wd->allocation.width;
  *height = (*height > 0)?*height:wd->allocation.height;
  
  /* We create a pixmap context and make this context current. */
  dumpData = visuOpenGLNew_pixmapContext((guint)*width, (guint)*height);
  /* We set the glViewport of this new context. */
  glViewport(0, 0, *width, *height);
  /* We call the given draw method. */
  render->redraw((const char**)0, render->redrawData);
  /* We copy the pixmap into generic data. */
  image = visuOpenGLGet_pixmapData((guint)*width, (guint)*height, hasAlpha);
  /* We free the pixmap context. */
  visuOpenGLFree_pixmapContext(dumpData);
  /* We change back the context to the current rendering area. */
  openGLWidgetSet_current(render, TRUE);

  return image;
}

/* OpenGL functions, OS dependent. */
#ifdef IMPL_GTKGLEXT
static void openGLWidgetSet_pixelFormat(OpenGLWidget *render)
{
  GdkScreen *screen;

  DBG_fprintf(stderr, "Gtk OpenGL (glext): choose visual for %p.\n", render);
  /* Get the screen. */
  if (GDK_IS_DRAWABLE(GTK_WIDGET(render)->window))
    screen = gdk_drawable_get_screen(GDK_DRAWABLE(GTK_WIDGET(render)->window));
  else
    screen = gdk_screen_get_default();
  DBG_fprintf(stderr, " | get a screen number %d.\n",
	      gdk_x11_screen_get_screen_number(screen));

  render->glconfig = visuOpenGLGet_glConfig(screen);
  DBG_fprintf(stderr, " | get a visual.\n");
}
static void openGLWidgetInit_context(OpenGLWidget *render, gboolean contextIsDirect)
{
  GtkWidget *wd;

  DBG_fprintf(stderr, "Gtk OpenGL (glext): create an OpenGL context (%d).\n",
	      contextIsDirect);

  wd = GTK_WIDGET(render);

  /* Create a GL area on the widget window. */
  render->glwindow = gdk_gl_window_new(render->glconfig, wd->window, NULL);

  /* Finaly, create the context. */
  if (contextIsDirect)
    {
      render->context = gdk_gl_context_new(GDK_GL_DRAWABLE(render->glwindow),
					   (GdkGLContext*)0, TRUE, GDK_GL_RGBA_TYPE);
      if (!render->context)
	{
	  g_warning("Can't create a direct rendering context, try an inderect one.\n");
	  render->context = gdk_gl_context_new(GDK_GL_DRAWABLE(render->glwindow),
					       (GdkGLContext*)0, FALSE,
					       GDK_GL_RGBA_TYPE);
	  render->isContextDirect = FALSE;
	}
    }
  else
    render->context = gdk_gl_context_new(GDK_GL_DRAWABLE(render->glwindow),
					 (GdkGLContext*)0, FALSE, GDK_GL_RGBA_TYPE);
  DBG_fprintf(stderr, " | create the context %p (%d).\n", (gpointer)render->context,
	      (int)render->isContextDirect);

  if (!render->context)
    {
      g_error("Cannot create a GtkGLExt context.\n");
    }
}
static void openGLWidgetFree_openGL(OpenGLWidget *render)
{
  g_return_if_fail(IS_OPENGL_WIDGET(render));

  DBG_fprintf(stderr, "Gtk OpenGL (glext): freeing context & window.\n");
  if (render->glwindow)
    gdk_gl_window_destroy(render->glwindow);
  if (render->context)
    gdk_gl_context_destroy(render->context);
}
gboolean openGLWidgetSet_current(OpenGLWidget *render, gboolean force)
{
  gboolean res;
  GtkWidget *wd;
  
  g_return_val_if_fail(IS_OPENGL_WIDGET(render), FALSE);

  if (!force && OPENGL_WIDGET_GET_CLASS(render)->contextCurrent == render)
    return TRUE;

  DBG_fprintf(stderr, "Gtk OpenGL (glext): %p is set current.\n", (gpointer)render);
  res = gdk_gl_drawable_make_current(GDK_GL_DRAWABLE(render->glwindow),
				     render->context);

  if (!res)
    {
      g_warning("Cannot make the openGLWidget object %p current.\n",
		(gpointer)render);
      return FALSE;
    }
  DBG_fprintf(stderr, "Gtk OpenGL (glext): OK context changed.\n");

  OPENGL_WIDGET_GET_CLASS(render)->contextCurrent = render;

  wd = GTK_WIDGET(render);
  openGLWidgetSet_viewport(render, wd->allocation.width, wd->allocation.height, FALSE);

  return TRUE;
}
void openGLWidgetSwap_buffers(OpenGLWidget *render)
{
  g_return_if_fail(OPENGL_WIDGET_GET_CLASS(render)->contextCurrent == render);

  DBG_fprintf(stderr, "Gtk OpenGL (glext): swap buffers of area %p.\n",
	      (gpointer)render);
  gdk_gl_drawable_swap_buffers(GDK_GL_DRAWABLE(render->glwindow));
}
static GdkColormap* openGLWidgetGet_openGLColormap(OpenGLWidget *render)
{
  g_return_val_if_fail(render->glconfig, (GdkColormap*)0);

  return gdk_gl_config_get_colormap(render->glconfig);
}
#endif
#ifdef IMPL_BUILTIN_X11
static void openGLWidgetSet_pixelFormat(OpenGLWidget *render)
{
  gint screenId;
  gboolean built;

  built = GDK_IS_DRAWABLE(GTK_WIDGET(render)->window);
  DBG_fprintf(stderr, "Gtk OpenGL (X11): choose visual for %p.\n", (gpointer)render);
  if (built)
    render->dpy = gdk_x11_drawable_get_xdisplay(GDK_DRAWABLE(GTK_WIDGET(render)->window));
  else
    render->dpy = gdk_x11_get_default_xdisplay();
  DBG_fprintf(stderr, " | get the display %p.\n", (gpointer)render->dpy);

  /* Get the screen id. */
  if (built)
    screenId = gdk_x11_screen_get_screen_number
      (gdk_drawable_get_screen(GDK_DRAWABLE(GTK_WIDGET(render)->window)));
  else
    screenId = gdk_x11_get_default_screen();
  DBG_fprintf(stderr, " | get a screen number %d.\n", screenId);

  render->vinfo = visuOpenGLGet_visualInfo(render->dpy, screenId);
  DBG_fprintf(stderr, " | get a visual %d.\n", (int)render->vinfo->visualid);
}
static void openGLWidgetInit_context(OpenGLWidget *render, gboolean contextIsDirect)
{
  GtkWidget *wd;

  DBG_fprintf(stderr, "Gtk OpenGL (X11): create an OpenGL context (%d).\n",
	      contextIsDirect);

  wd = GTK_WIDGET(render);
  /* Check for GLX. */
  if (!glXQueryExtension(render->dpy, 0, 0))
    {
      g_error("No GLX extension.\nYour X server"
	      " does not support OpenGL extension. Please contact your"
	      " system administrator to ask him to add the 'glx'"
	      " extension to your X server.\n");
    }

  /* Finaly, create the context. */
  if (contextIsDirect)
    {
      render->context = glXCreateContext(render->dpy, render->vinfo, 0, GL_TRUE);
      if (!render->context)
	{
	  g_warning("Can't create a direct rendering context, try an inderect one.\n");
	  render->context = glXCreateContext(render->dpy, render->vinfo, 0, GL_FALSE);
	  render->isContextDirect = FALSE;
	}
    }
  else
    render->context = glXCreateContext(render->dpy, render->vinfo, 0, GL_FALSE);
  DBG_fprintf(stderr, " | create the context %p (%d).\n", (gpointer)render->context,
	    (int)render->isContextDirect);

  if (!render->context)
    {
      g_error("Cannot create a GLX context.\n");
    }
}
static void openGLWidgetFree_openGL(OpenGLWidget *render)
{
  g_return_if_fail(IS_OPENGL_WIDGET(render));

  if (render->dpy)
    {
      DBG_fprintf(stderr, "Free (X11): freeing context.\n");
      if (render->context)
	glXDestroyContext(render->dpy, render->context);
      /* We do NOT close the display since it is shared
	 by all the application and thus dpy structure
	 is unique and will be closed by GTK when quiting. */
/*       XCloseDisplay(render->dpy); */
    }
}
gboolean openGLWidgetSet_current(OpenGLWidget *render, gboolean force)
{
  int res;
  GtkWidget *wd;
  XID windowId;
  
  g_return_val_if_fail(IS_OPENGL_WIDGET(render), FALSE);

  if (!force && OPENGL_WIDGET_GET_CLASS(render)->contextCurrent == render)
    return TRUE;

  DBG_fprintf(stderr, "Gtk OpenGL (X11): widget visualID %d.\n",
	      (int)gdk_x11_visual_get_xvisual
	      (gtk_widget_get_visual(GTK_WIDGET(render)))->visualid);

  DBG_fprintf(stderr, "Gtk OpenGL (X11): %p is set current.\n", (gpointer)render);

  windowId = gdk_x11_drawable_get_xid(GDK_DRAWABLE(GTK_WIDGET(render)->window));
  DBG_fprintf(stderr, "Gtk OpenGL (X11): widget windowID %d.\n", (int)windowId);

  res = glXMakeCurrent(render->dpy, (GLXDrawable)windowId,
		       render->context);
  if (!res)
    {
      g_warning("Cannot make the openGLWidget object %p current.\n",
		(gpointer)render);
      return FALSE;
    }
  DBG_fprintf(stderr, "Gtk OpenGL (X11): OK context changed.\n");

  /* Now that the glx tunnel has been added, we need
     to specify again that we want a backing store because until now
     the backing store is only for the X window (and thus is black)
     but not for the glx screen. */
/*   wattrs.backing_store = Always; */
/*   XChangeWindowAttributes(render->dpy, windowId, */
/* 			  CWBackingStore, &wattrs); */

  OPENGL_WIDGET_GET_CLASS(render)->contextCurrent = render;

  wd = GTK_WIDGET(render);
  openGLWidgetSet_viewport(render, wd->allocation.width, wd->allocation.height, FALSE);

  return TRUE;
}
void openGLWidgetSwap_buffers(OpenGLWidget *render)
{
  g_return_if_fail(OPENGL_WIDGET_GET_CLASS(render)->contextCurrent == render);

  DBG_fprintf(stderr, "Gtk OpenGL (X11): swap buffers of area %p.\n", (gpointer)render);
  glXSwapBuffers(render->dpy, (GLXDrawable)gdk_x11_drawable_get_xid
		 (GDK_DRAWABLE(GTK_WIDGET(render)->window)));
}
static GdkColormap* openGLWidgetGet_openGLColormap(OpenGLWidget *render)
{
  g_return_val_if_fail(IS_OPENGL_WIDGET(render), (GdkColormap*)0);
  g_return_val_if_fail(render->vinfo, (GdkColormap*)0);

  return gdk_colormap_new(gdkx_visual_get(render->vinfo->visualid), FALSE);
}
#endif
#ifdef IMPL_BUILTIN_WIN32
static void openGLWidgetSet_pixelFormat(OpenGLWidget *render)
{
  DBG_fprintf(stderr, "Gtk OpenGL (Win32): choose visual for %p.\n", render);

  render->windowId = (HWND)gdk_win32_drawable_get_handle(GDK_DRAWABLE(GTK_WIDGET(render)->window));
  render->hdc = GetDC(render->windowId);
  DBG_fprintf(stderr, " | get the hdc %d.\n", (int)render->hdc);

  visuOpenGLSetup_pixelFormat(render->hdc);
}
static void openGLWidgetInit_context(OpenGLWidget *render, gboolean contextIsDirect)
{
  DBG_fprintf(stderr, "Gtk OpenGL (Win32): create an OpenGL context (%d).\n", contextIsDirect);

  /* Finaly, create the context. */
  render->context = wglCreateContext(render->hdc);
}
static void openGLWidgetFree_openGL(OpenGLWidget *render)
{
  g_return_if_fail(IS_OPENGL_WIDGET(render));

  if (render->context)
    wglDeleteContext(render->context);
/*   if (render->hdc) */
/*     DeleteDC(render->hdc); */
}
gboolean openGLWidgetSet_current(OpenGLWidget *render, gboolean force _U_)
{
  GtkWidget *wd;

  g_return_val_if_fail(IS_OPENGL_WIDGET(render), FALSE);

  DBG_fprintf(stderr, "Gtk OpenGL (action) : %p is set current.\n", (gpointer)render);
  wglMakeCurrent(NULL, NULL);
  wglMakeCurrent(render->hdc, render->context);

  OPENGL_WIDGET_GET_CLASS(render)->contextCurrent = render;

  wd = GTK_WIDGET(render);
  openGLWidgetSet_viewport(render, wd->allocation.width, wd->allocation.height, FALSE);

  return TRUE;
}
void openGLWidgetSwap_buffers(OpenGLWidget *render)
{
  g_return_if_fail(OPENGL_WIDGET_GET_CLASS(render)->contextCurrent == render);

  DBG_fprintf(stderr, "Gtk OpenGL (action) : swap buffers of area %p.\n", (gpointer)render);
  SwapBuffers(render->hdc);
}
static GdkColormap* openGLWidgetGet_openGLColormap(OpenGLWidget *render)
{
  g_return_val_if_fail(IS_OPENGL_WIDGET(render), (GdkColormap*)0);
  
  return gdk_screen_get_system_colormap(gdk_screen_get_default());
}
#endif
