
##--------------------------------------------------------------------##
##--- The core dispatch loop, for jumping to a code address.       ---##
##---                                               x86/dispatch.S ---##
##--------------------------------------------------------------------##

/*
  This file is part of Valgrind, an extensible x86 protected-mode
  emulator for monitoring program execution on x86-Unixes.

  Copyright (C) 2000-2005 Julian Seward 
     jseward@acm.org

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
  02111-1307, USA.

  The GNU General Public License is contained in the file COPYING.
*/

#include "core_asm.h"
#include "x86_private_asm.h"


/*------------------------------------------------------------*/
/*--- The normal-case dispatch machinery.                  ---*/
/*------------------------------------------------------------*/
	
/* To transfer to an (original) code address, load it into %eax and
   jump to vg_dispatch.  This fragment of code tries to find the
   address of the corresponding translation by searching the translation
   table.   If it fails, a new translation is made, added to the
   translation table, and then jumped to.  Almost all the hard
   work is done by C routines; this code simply handles the
   common case fast -- when the translation address is found in
   the translation cache.

   At entry, %eax is the only live (real-machine) register; the
   entire simulated state is tidily saved in vg_m_state.  
*/

	
#define TT_LOOKUP(reg, fail)				\
	movl %eax, reg;					\
	andl $VG_TT_FAST_MASK, reg;			\
	movl VG_(tt_fast)(,reg,4), reg;			\
	cmpl %eax, (reg);				\
	jne  fail;					\
	addl $VG_CODE_OFFSET, reg
	
/* The C world needs a way to get started simulating.  So we provide
   a function void vg_run_innerloop (), which starts running
   from vg_m_eip, and exits when the counter reaches zero.  This loop
   can also exit if it catches a synchronous signal, for example SIGSEGV.
   It then longjmp()s back past here.

   Prototype:	
	UInt VG_(run_innerloop)(ThreadState *tst);
*/

.globl VG_(run_innerloop)
VG_(run_innerloop):
	/* OYNK(1000) */

	/* ----- entry point to VG_(run_innerloop) ----- */
#define FSZ	(5*4 + 4)	/* 5 pushes + retaddr */
	pushl	%ebx
	pushl	%esi
	pushl	%edi
	pushl	%ebp
	pushl	%fs						/* always save %fs */

	/* check to see if we're doing pointer checking */
	cmpb	$0, VG_(clo_pointercheck)
	jz	1f
	
	mov	$(VG_POINTERCHECK_SEGIDX << 3) + 7, %eax	/* load new %fs */
	movw	%ax,%fs

	/* Set up the ThreadState pointer */
1:	movl	FSZ(%esp), %ebp

	/* fetch m_eip into %eax */
	movl	VGOFF_eip(%ebp), %eax
	
	/* Jump here to do a new dispatch.
	   %eax holds destination (original) address.
	   %ebp indicates further details of the control transfer
	   requested to the address in %eax.
	
	   If ebp == tst, just jump next to %eax.
	 
	   If ebp == VG_EBP_JMP_SYSCALL, do a system call before 
	   continuing at eax.
	
	   If ebp == VG_EBP_JMP_CLIENTREQ, do a client request before 
	   continuing at eax.
	
	   If %ebp has any other value, we panic.
	*/

	/* try a fast lookup in the translation cache */
2:	TT_LOOKUP(%ebx, fast_lookup_failed)

	/* Found a match.  Call the tce.payload field */
	incl	VG_(unchained_jumps_done)	      /* update stats */

	/* call generated code */
	call	*%ebx

	movl	FSZ(%esp), %ebx			/* ebx==ThreadState */

	/* save the jump address at tst->arch.m_eip */
	movl	%eax, VGOFF_eip(%ebx)
	
	/* Are we out of timeslice?  If yes, defer to scheduler. */
	cmpl	$0, VGOFF_DISPATCH_CTR(%ebx)
	jz	counter_is_zero

	cmpl	%ebx, %ebp
	jz	2b		/* thread still current	;  keep going */

	movl	%ebp, %eax	/* return exceptional dispatch */

3:	popl	%fs
	popl	%ebp
	popl	%edi
	popl	%esi
	popl	%ebx
	ret	
	
fast_lookup_failed:
	/* %EIP is up to date here since dispatch_boring dominates */
	movl	$VG_TRC_INNER_FASTMISS, %eax
	jmp	3b

counter_is_zero:
	/* %EIP is up to date here since dispatch_boring dominates */
	movl	$VG_TRC_INNER_COUNTERZERO, %eax
	jmp	3b
	

/*
	This is the translation chainer, our run-time linker, if you like.

	VG_(patch_me) patches the call instruction in the jump site
	with a jump to the generated code for the branch target.  %eax
	contains the original program's EIP - if we get a hit in
	tt_fast, then the call is patched into a jump; otherwise it
	simply drops back into the dispatch loop for normal
	processing.

	The callsite is expected to look like:
		call	VG_(patch_me)
	it will be transformed into
		jmp	$TARGETADDR

	The environment we're expecting on entry is:
		%eax    = branch target address (original code EIP)
		*(%esp) = just after call
*/
.globl VG_(patch_me)
VG_(patch_me):
	/* try a fast lookup in the translation cache */
	TT_LOOKUP(%ebx, 1f)

	/* ebx = target eip */

	/* Patch call instruction at callsite into a chained jmp */
	popl	%ecx	    /* ecx = just after (VG_PATCHME_CALLSZ byte) call */
	subl	%ecx, %ebx		/* ebx = delta */
	movb	$0xE9, -(VG_PATCHME_CALLSZ-0)(%ecx)		/* 0xe9 = jmp */
	movl	%ebx,  -(VG_PATCHME_CALLSZ-1)(%ecx)	       /* store delta */
	addl	%ecx, %ebx
	incl	VG_(bb_enchain_count)			      /* update stats */
	jmp	*%ebx					       /* jmp to dest */

	/* tt_fast miss: return into main dispatch loop */
1:	addl	$4, %esp	/* remove our call address */
	ret			/* return into main dispatch loop above */

/* This is called from generated code when we're about to 
   start running a wrapped function.  When called, the only live
   registers are %eax and %ebp, so they're the only ones we preserve.

   We are called with FuncWrapper * on the stack
	
   This calls VG_(wrap_before)(ThreadState *tst, FuncWrapper *wrapper)
*/
.globl VG_(helper_wrapper_before)
VG_(helper_wrapper_before):	
	pushl	%eax		/* save %eax */
	pushl	%ebp		/* save %ebp */
	
	pushl	12(%esp)	/* pass FuncWrapper * */
	pushl	%ebp		/* pass %ebp */
	call	VG_(wrap_before)
	add	$8, %esp	/* remove args */
	
	popl	%ebp		/* restore %ebp */
	popl	%eax		/* restore %eax */
	ret	$4		/* remove argument */

/* This is called from generated code when we're returning from a wrapped function.
   Again, the only live registers are %eax and %ebp.
   This  is inserted instead of the normal dispatch_ctr management code, so we need
   to duplicate that functionality too.
*/
.globl VG_(helper_wrapper_return)
VG_(helper_wrapper_return):
	subl	$1, VGOFF_DISPATCH_CTR(%ebp)
	jz	1f
	pushl	%eax
	pushl	%ebp
	call	VG_(wrap_after)
	popl	%ebp
	popl	%eax
	ret
1:	/* counter zero */
	add	$4, %esp	/* remove return addr */
	ret			/* return back into dispatch loop */
	
.data
panic_msg_ebp:
.ascii	"vg_dispatch: %ebp has invalid value!"
.byte	0
.text	
        
/* Let the linker know we don't need an executable stack */
.section .note.GNU-stack,"",@progbits

##--------------------------------------------------------------------##
##--- end                                                          ---##
##--------------------------------------------------------------------##
