/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef FOLDERWATCHPROVIDER_H
#define FOLDERWATCHPROVIDER_H

#include <QObject>
#include <QList>
#include <QMap>

class QTreeWidget;
class QTreeWidgetItem;

class ColumnConditionSettings;
class IssueRowFilter;
class ProjectRow;
class FolderRow;

/**
* Class providing folder watches to the FolderWatchWriter.
*
* This class calculates the status of folder watches, formats it into localized text
* and outputs it to the FolderWatchWriter.
*/
class FolderWatchProvider : public QObject
{
    Q_OBJECT
public:
    /**
    * Default constructor.
    */
    FolderWatchProvider( QObject* parent );

    /**
    * Destructor.
    */
    ~FolderWatchProvider();

public:
    enum {
        FolderIdRole = Qt::UserRole + 10,
        FilterNameRole
    };

public:
    /**
    * Calculate watches for all folders.
    */
    void updateAll();

    /**
    * Calculate watches for all folders of the given type.
    */
    void updateFolders( int typeId );

    /**
    * Calculate watches of a single folder.
    */
    void updateFolder( int folderId );

    /**
    * Return @c true if there are no watches.
    */
    bool isEmpty();

    /**
    * Setup columns in the tree widget.
    */
    void setupColumns( QTreeWidget* tree );

    /**
    * Populate the status of the watches to the given tree widget.
    */
    void populateTree( QTreeWidget* tree );

private:
    void calculateWatches( int folderId, ColumnConditionSettings* settings );

    QTreeWidgetItem* addProjectItem( QTreeWidget* tree, const ProjectRow* project );
    QTreeWidgetItem* addFolderItem( QTreeWidgetItem* parent, const FolderRow* folder );
    QTreeWidgetItem* addFilterItem( QTreeWidgetItem* parent, int folderId, const QString& filter );

    void setCount( QTreeWidgetItem* item, int column, int count );

private:
    struct Watch
    {
        QString m_filter;
        int m_total;
        int m_new;
        int m_updated;

        static bool compare( const Watch& w1, const Watch& w2 );
    };

    typedef QList<Watch> WatchList;

private:
    QMap<int, WatchList> m_map;

    IssueRowFilter* m_filter;
};

#endif
