/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef LINKLOCATOR_H
#define LINKLOCATOR_H

#include <QString>

class QRegExp;

/**
* Text to HTML converter creating links for URLs and items.
*
* URLs and e-mail addresses are converted to hyperlinks. Issue, comment
* and attachments identifiers prefixed with a # character are converted
* to links using the <tt>id://{number}</tt> URL.
*
* If the PreserverSpaces flag is used, whitespaces are converted
* to non-collapsible spaces and HTML line breaks to preserve text formatting.
*
* Do not create instances of this class, use the convertToHtml() static
* method instead.
*/
class LinkLocator
{
public:
    /**
    * Formatting flags.
    */
    enum Flags
    {
        /**
        * Do not collapse whitespaces.
        */
        PreserveSpaces = 1,
        /**
        * Use anchors instead of internal id:// links.
        */
        NoInternalLinks = 2
    };

protected:
    /**
    * Constructor.
    *
    * @param reserve Initial size of the buffer to allocate.
    *
    * @param flags Conversion flags.
    */
    LinkLocator( int reserve, int flags );

    /**
    * Destructor.
    */
    ~LinkLocator();

protected:
    /**
    * Append a fragment of text to the buffer.
    */
    void append( const QString& text );

    /**
    * Append a fragment of raw HTML to the buffer.
    */
    void appendTag( const QString& tag );

    /**
    * Return the content of the buffer.
    */
    const QString& result() const { return m_result; }

public:
    /**
    * Convert given text to HTML.
    *
    * @param text Plain text to convert.
    *
    * @param flags Formatting flags.
    *
    * @return The converted text.
    */
    static QString convertToHtml( const QString& text, int flags );

    /**
    * Escape HTML entities in the given text.
    */
    static QString escape( const QString& text );

private:
    QString m_result;

    int m_flags;

    int m_column;
    bool m_space;
};

#endif
