/*  xxGlobe   Global location chooser.
    Last modified 1998-05-06

    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "xtide.hh"

// This is my best guess as to how to do the Orthographic Projection.
// Returns 1 if you should draw it, 0 if skip.
int
xxGlobe::translateCoordinates (Coordinates c_in, int *x, int *y,
double my_center_longitude) {
  if (c_in.isNull())
    return 0;
  double lng = c_in.lng();

  // All of this is just to get the longitude coordinated with the center
  // given that it wraps around.
  if (lng < my_center_longitude - 90 || lng > my_center_longitude + 90) {
    lng += 360.0;
    if (lng < my_center_longitude - 90 || lng > my_center_longitude + 90) {
      lng -= 360.0;
      lng -= 360.0;
      if (lng < my_center_longitude - 90 || lng > my_center_longitude + 90)
        return 0;
    }
  }
  lng -= my_center_longitude;

  double height = -sin (Degrees (c_in.lat()));
  *y = (int) (height * size / 2 + size / 2);
  double width = sqrt (1.0 - height*height) * size / 2;
  *x = (int) (size / 2 + (sin (Degrees (lng)) * width));
  return 1;
}

// If you support zooming and panning by drawing the globe bigger and
// parenting it with a viewport, you run out of memory VERY fast.
// Zoom 10x = 5000x5000x3 bytes = 75 MEGS

void
xxGlobe::redrawGlobe () {

  if (xorigin < 0)
    xorigin = 0;
  else if (xorigin > (int)size - min_globe_size)
    xorigin = (int)size - min_globe_size;

  if (yorigin < 0)
    yorigin = 0;
  else if (yorigin > (int)size - min_globe_size)
    yorigin = (int)size - min_globe_size;

  XFillRectangle (picture->display, globepixmap,
    picture->background_gc, 0, 0, min_globe_size+1, min_globe_size+1);
  int latitude, longitude;
  for (longitude = -90; longitude <= -30; longitude += 30) {
    int x, y;
    assert (translateCoordinates (Coordinates (0.0, (double)longitude), &x, &y,
      0.0));
    unsigned width = 2 * (size / 2 - x);
    XDrawArc (picture->display, globepixmap, picture->text_gc,
      x-xorigin, -yorigin, width, size, 0, 360*64);
  }
  // Remember, (0,0) is considered null and won't do translateCoordinates.
  XDrawLine (picture->display, globepixmap, picture->text_gc,
    size/2-xorigin, -yorigin, size/2-xorigin, size-yorigin);
  for (latitude = -60; latitude <= 60; latitude += 30) {
    int x, y;
    assert (translateCoordinates (Coordinates ((double)latitude, -90.0), &x,
      &y, 0.0));
    unsigned width = 2 * (size / 2 - x);
    XDrawLine (picture->display, globepixmap, picture->text_gc,
      x-xorigin, y-yorigin, x+width-xorigin, y-yorigin);
  }
  unsigned long i;
  StationIndex *si = new StationIndex();
  for (i=0; i<stationIndex->length(); i++) {
    int x, y;
    if (translateCoordinates ((*stationIndex)[i]->coordinates, &x, &y,
			      center_longitude)) {
      int ox = x-xorigin;
      int oy = y-yorigin;
      if (ox >= -1 && ox <= min_globe_size && oy >= -1 && oy <= min_globe_size) {
        XFillArc (picture->display, globepixmap, picture->mark_gc,
          ox-1, oy-1, 3, 3, 0, 360*64);
        si->add ((*stationIndex)[i]);
      }
    }
  }

  Arg args[1] = {
    {"bitmap", (XtArgVal)globepixmap}
  };
  XtSetValues (picture->manager, args, 1);
  picture->refresh();
  blastflag = 0;
  locationlist->changeList (si);
}

static void
rotateCallback (Widget w, XtPointer client_data, XtPointer call_data) {
  xxGlobe *globe = (xxGlobe *)client_data;
  globe->center_longitude -= 30;
  if (globe->center_longitude < -180)
    globe->center_longitude += 360;
  globe->redrawGlobe();
}

static void
dismissCallback (Widget w, XtPointer client_data, XtPointer call_data) {
  xxGlobe *globe = (xxGlobe *)client_data;
  globe->dismiss();
}

void
xxGlobeHelpCallback (Widget w, XtPointer client_data, XtPointer call_data) {
  xxGlobe *globe = (xxGlobe *)client_data;
  Dstr helpstring ("\
XTide Location Chooser -- Globe Window\n\
\n\
The globe window initially shows an entire hemisphere.  Tide stations\n\
in that hemisphere are projected onto the globe as red dots and\n\
enumerated in the location list window.  Other tide stations can be\n\
accessed by rotating the globe.\n\
\n\
Mouse buttons:\n\
\n\
  Left:  zoom in on the clicked region and narrow the location list.\n\
  You can zoom in 6 times for a maximum 64x magnification, after which\n\
  the left mouse button behaves just like the right mouse button.\n\
\n\
  Right:  narrow the location list to the clicked area.  A circle will\n\
  be drawn on the globe showing the radius included, but no zooming\n\
  will occur.\n\
\n\
  Middle:  select a tide station.  (You can also select a tide station\n\
  by left-clicking on it in the location list.)\n\
\n\
Buttons in the globe window:\n\
\n\
  Rotate:  turn the globe 30 degrees and update the location list\n\
  accordingly.\n\
\n\
  List All:  include all available tide stations in the location list,\n\
  even those whose latitude and longitude are unknown (null).\n\
\n\
  Zoom Out:  self-explanatory.  Sufficient usage will return to the\n\
  hemisphere view.\n\
\n\
  Dismiss:  remove the location chooser.  Any windows containing tide\n\
  predictions will remain.\n\
\n\
Keyboard:\n\
\n\
  Arrow keys:  pan up/down/left/right.");
  globe->xtidecontext->root->newHelpBox (helpstring);
}

void
listAllCallback (Widget w, XtPointer client_data, XtPointer call_data) {
  xxGlobe *globe = (xxGlobe *)client_data;
  globe->locationlist->changeList (globe->stationIndex->clone());
}

void
zoomOutCallback (Widget w, XtPointer client_data, XtPointer call_data) {
  xxGlobe *globe = (xxGlobe *)client_data;
  if (globe->size == min_globe_size)
    return;
  if (globe->size / zoomfactor > min_globe_size) {
    double cx = (double)(globe->xorigin + min_globe_size/2) / (double)globe->size;
    double cy = (double)(globe->yorigin + min_globe_size/2) / (double)globe->size;
    globe->size = (unsigned)(globe->size / zoomfactor);
    globe->xorigin = (int)(cx * globe->size - min_globe_size / 2);
    globe->yorigin = (int)(cy * globe->size - min_globe_size / 2);
  } else {
    globe->size = min_globe_size;
    globe->xorigin = globe->yorigin = 0;
  }
  globe->redrawGlobe();
}

// This is done by the flat window image and not by spherical projection.
// It's simpler this way.
void
xxGlobe::blast (int x, int y) {
  if (blastflag)
    XDrawArc (picture->display, globepixmap, picture->invert_gc, blastx,
      blasty, blastradius*2, blastradius*2, 0, 360*64);

  blastflag = 1;
  blastx = x - blastradius;
  blasty = y - blastradius;
  XDrawArc (picture->display, globepixmap, picture->invert_gc, blastx,
    blasty, blastradius*2, blastradius*2, 0, 360*64);

  Arg args[1] = {
    {"bitmap", (XtArgVal)globepixmap}
  };
  XtSetValues (picture->manager, args, 1);
  picture->refresh();

  // Make a list of all locations within the blast radius.
  int bx = x + xorigin;
  int by = y + yorigin;
  unsigned long a;
  StationIndex *si = new StationIndex();
  for (a=0; a<stationIndex->length(); a++) {
    int x, y;
    if (translateCoordinates ((*(stationIndex))[a]->coordinates,
      &x, &y, center_longitude)) {
      double dx = (double)x - (double)bx;
      double dy = (double)y - (double)by;
      if (sqrt (dx*dx + dy*dy) <= blastradius)
	si->add ((*(stationIndex))[a]);
    }
  }
  locationlist->changeList (si);
}

void xxGlobeKeyboardEventHandler (Widget w, XtPointer client_data,
				  XEvent *event, Boolean *continue_dispatch) {
  xxGlobe *globe = (xxGlobe *)client_data;
  XKeyEvent *xke = (XKeyEvent *)event;
  KeySym foo = XLookupKeysym (xke, 0); // Index 0 = no shift/ctrl/meta/etc.
  switch (foo) {
  case XK_Left:
  case XK_KP_Left:
    globe->xorigin -= min_globe_size/2;
    break;
  case XK_Up:
  case XK_KP_Up:
    globe->yorigin -= min_globe_size/2;
    break;
  case XK_Right:
  case XK_KP_Right:
    globe->xorigin += min_globe_size/2;
    break;
  case XK_Down:
  case XK_KP_Down:
    globe->yorigin += min_globe_size/2;
    break;
  default:
    return;
  }
  globe->redrawGlobe();
}

void
xxGlobeButtonPressEventHandler (Widget w, XtPointer client_data,
			 XEvent *event, Boolean *continue_dispatch) {
  xxGlobe *globe = (xxGlobe *)client_data;
  XButtonEvent *xbe = (XButtonEvent *)event;

  // Coordinates relative to upper left corner of globe pixmap.
  // Need to subtract the internal margin of the label widget.
  int rx = xbe->x - globe->internalWidth;
  int ry = xbe->y - globe->internalHeight;

  // Relative to entire globe.
  int bx = rx + globe->xorigin;
  int by = ry + globe->yorigin;

  if (xbe->button == Button3 ||
      (xbe->button == Button1 && globe->size == max_globe_size)) {
    globe->blast (rx, ry);
    return;
  }

  // Zoom
  if (xbe->button == Button1) {
    double cx = (double)bx / (double)globe->size;
    double cy = (double)by / (double)globe->size;
    if (globe->size * zoomfactor < max_globe_size)
      globe->size = (unsigned)(zoomfactor * globe->size);
    else
      globe->size = max_globe_size;
    globe->xorigin = (int)(cx * globe->size - min_globe_size / 2);
    globe->yorigin = (int)(cy * globe->size - min_globe_size / 2);
    globe->redrawGlobe();
    return;
  }

  // Load location
  if (xbe->button == Button2) {
    // Find nearest location that is close enough (4 pixels).
    unsigned long a;
    StationRef *closestsr = NULL;
    double d = 17.0;
    for (a=0; a<globe->stationIndex->length(); a++) {
      int x, y;
      if (globe->translateCoordinates ((*(globe->stationIndex))[a]->coordinates,
	&x, &y, globe->center_longitude)) {
	double dx = (double)x - (double)bx;
	double dy = (double)y - (double)by;
        double dd = dx*dx + dy*dy;
        if (dd < d) {
          d = dd;
          closestsr = (*(globe->stationIndex))[a];
        }
      }
    }
    if (closestsr)
      globe->xtidecontext->root->newGraph (closestsr);
  }

  // X11 has Button4 and Button5... go figure.
}

void
xxGlobe::dismiss () {
  delete this; // Is this safe?
}

xxGlobe::xxGlobe (xxContext *context, StationIndex &in_stationIndex,
xxTideContext *in_tidecontext): xxWindow (in_tidecontext, context, 1) {
  mypopup->setTitle ("Globe");
  size = min_globe_size;
  xorigin = yorigin = blastflag = 0;
  stationIndex = &in_stationIndex;

  globepixmap = mypopup->makePixmap (min_globe_size+1, min_globe_size+1);
  Arg args[3] = {
    {XtNbitmap, (XtArgVal)globepixmap},
    {XtNbackground, (XtArgVal)mypopup->pixels[Colors::background]},
    {XtNforeground, (XtArgVal)mypopup->pixels[Colors::foreground]}
  };
  Widget picturewidget = XtCreateManagedWidget ("", labelWidgetClass,
    box->manager, args, 3);
  picture = new xxContext (box, picturewidget);
  XtAddEventHandler (picturewidget, ButtonPressMask, False,
    xxGlobeButtonPressEventHandler, (XtPointer)this);
  XtAddEventHandler (picturewidget, KeyPressMask, False,
    xxGlobeKeyboardEventHandler, (XtPointer)this);

  {
    Arg args[2] = {
      {XtNinternalHeight, (XtArgVal)(&internalHeight)},
      {XtNinternalWidth, (XtArgVal)(&internalWidth)}
    };
    XtGetValues (picture->manager, args, 2);
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
    cerr << "Globe internal height " << internalHeight <<
      " internal width " << internalWidth << endl;
#endif
  }

  Arg buttonargs[2] =  {
    {XtNbackground, (XtArgVal)mypopup->pixels[Colors::button]},
    {XtNforeground, (XtArgVal)mypopup->pixels[Colors::foreground]}
  };

  {
    Widget buttonwidget = XtCreateManagedWidget ("Rotate", repeaterWidgetClass,
      box->manager, buttonargs, 2);
    XtAddCallback (buttonwidget, XtNcallback, rotateCallback, (XtPointer)this);
    rotatebutton = new xxContext (mypopup, buttonwidget);
  }{
    Widget buttonwidget = XtCreateManagedWidget ("List All",
      commandWidgetClass, box->manager, buttonargs, 2);
    XtAddCallback (buttonwidget, XtNcallback, listAllCallback,
      (XtPointer)this);
    listAllButton = new xxContext (mypopup, buttonwidget);
  }{
    Widget buttonwidget = XtCreateManagedWidget ("Zoom Out",
      commandWidgetClass, box->manager, buttonargs, 2);
    XtAddCallback (buttonwidget, XtNcallback, zoomOutCallback,
      (XtPointer)this);
    zoomoutbutton = new xxContext (mypopup, buttonwidget);
  }{
    Widget buttonwidget = XtCreateManagedWidget ("Dismiss", commandWidgetClass,
      box->manager, buttonargs, 2);
    XtAddCallback (buttonwidget, XtNcallback, dismissCallback,
      (XtPointer)this);
    dismissbutton = new xxContext (mypopup, buttonwidget);
  }{
    Widget buttonwidget = XtCreateManagedWidget ("?", commandWidgetClass,
      box->manager, buttonargs, 2);
    XtAddCallback (buttonwidget, XtNcallback, xxGlobeHelpCallback,
      (XtPointer)this);
    helpbutton = new xxContext (mypopup, buttonwidget);
  }

  mypopup->realize();
  mypopup->fixSize();

  Settings *settings = in_tidecontext->settings;
  if (settings->gl_isnull != 0 || settings->gl == 360.0)
    center_longitude = stationIndex->bestCenterLongitude();
  else
    center_longitude = settings->gl;

  // Start with an empty list since it's going to get clobbered in
  // redrawGlobe anyway.
  locationlist = new xxLocationList (mypopup, new StationIndex(),
    xtidecontext, this);
  redrawGlobe();
}

xxGlobe::~xxGlobe () {
  mypopup->unrealize();
  delete locationlist;
  delete dismissbutton;
  delete rotatebutton;
  delete helpbutton;
  delete listAllButton;
  delete picture;
  XFreePixmap (mypopup->display, globepixmap);
}

void xxGlobe::global_redraw() {
  xxWindow::global_redraw();
  redrawGlobe();
  Arg buttonargs[2] =  {
    {XtNbackground, (XtArgVal)mypopup->pixels[Colors::button]},
    {XtNforeground, (XtArgVal)mypopup->pixels[Colors::foreground]}
  };
  if (rotatebutton)
    XtSetValues (rotatebutton->manager, buttonargs, 2);
  if (dismissbutton)
    XtSetValues (dismissbutton->manager, buttonargs, 2);
  if (listAllButton)
    XtSetValues (listAllButton->manager, buttonargs, 2);
  if (helpbutton)
    XtSetValues (helpbutton->manager, buttonargs, 2);
  if (zoomoutbutton)
    XtSetValues (zoomoutbutton->manager, buttonargs, 2);
  Arg args[2] =  {
    {XtNbackground, (XtArgVal)mypopup->pixels[Colors::background]},
    {XtNforeground, (XtArgVal)mypopup->pixels[Colors::foreground]}
  };
  if (picture)
    XtSetValues (picture->manager, args, 2);
}
