/*  TideDialog class.  */

/*****************************************************************************\

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

\*****************************************************************************/



#include "tideDialog.h"
#include "tideDialogHelp.h"



//  This is the alpha sort function for qsort.

static int compare_names (const void *a, const void *b)
{
    char low_left[15], low_right[15];


    CONSTITUENT left = *(CONSTITUENT *)(a);
    CONSTITUENT right = *(CONSTITUENT *)(b);

    for (int i = 0 ; i < strlen (left.name) ; i++)
        low_left[i] = tolower (left.name[i]);
    low_left[strlen (left.name)] = 0;

    for (int i = 0 ; i < strlen (right.name) ; i++)
        low_right[i] = tolower (right.name[i]);
    low_right[strlen (right.name)] = 0;

    return (strcmp (low_left, low_right));
}





//  If the TIDE_RECORD, lrec, is entered with a header.record_number less than
//  zero we will be adding a new record.

TideDialog::TideDialog (QWidget *parent, const char *name, TIDE_RECORD lrec, 
                        TideEditor *te):
  QTabDialog (parent, name)
{
    int       i;


    db = get_tide_db_header ();
    rec = lrec;
    tideEditor = te;

    sub_station = NULL;


    cnst = (CONSTITUENT *) calloc (db.constituents, sizeof (CONSTITUENT));
    if (cnst == NULL)
      {
        perror ("Allocating constituent list");
        exit (-1);
      }


    for (i = 0 ; i < db.constituents ; i++)
    {
        cnst[i].num = i;
        strcpy (cnst[i].name, get_constituent (i));
    }


    //  Sort the names then update the constituent list.

    ::qsort (cnst, db.constituents, sizeof (CONSTITUENT), &compare_names);


    tideDialog = new QTabDialog (this, 0, TRUE, WDestructiveClose);
    tideDialog->resize (640, 640);
    tideDialog->setCaption (name);
    tideDialog->setCancelButton ("Cancel");
    tideDialog->setHelpButton ("Print");

    connect (tideDialog, SIGNAL (applyButtonPressed ()), this, 
        SLOT (slotUpdateStation ()));
    connect (tideDialog, SIGNAL (helpButtonPressed ()), this, 
        SLOT (slotPrintStation ()));
    connect (tideDialog, SIGNAL (cancelButtonPressed ()), this, 
        SLOT (slotCancelStation ()));


    if (rec.header.record_number >= 0)
    {
        tideDialog->setDefaultButton ("Delete Record");
        connect (tideDialog, SIGNAL (defaultButtonPressed ()), this, 
            SLOT (slotDeleteStation ()));
    }


    //  Build the normal tabs.

    generalTab ();
    constituentsTab ();
    offsetsTab ();

    QWhatsThis::add (tideDialog, tideDialogText);

    if (QApplication::desktop ()->width () > (640 + 50)
        && QApplication::desktop ()->height () > (640 + 66))
    {
        tideDialog->show ();
    }
    else
    {
        tideDialog->showMaximized ();
    }
}




TideDialog::~TideDialog ()
{
    if (cnst) free (cnst);
}



//  Information that is common to both Reference and Subordinate stations.

void
TideDialog::generalTab ()
{
    char hem;
    int i;
    double deg, min, sec;


    QVBox *generalTab = new QVBox (this);
    generalTab->setMargin (5);
    generalTab->setSpacing (5);

    QString rcnm = QString ("%1").arg (rec.header.record_number);
    if (rec.header.record_number == -1) rcnm = "";
    QLabel *reclab = new QLabel ("Record Number:", generalTab);
    QLabel *recnum = new QLabel (rcnm, generalTab);
    recnum->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    QWhatsThis::add (reclab, recnumText);
    QWhatsThis::add (recnum, recnumText);


    if (rec.header.reference_station >= 0)
    {
        QString rfnm (get_station (rec.header.reference_station));
        QLabel *reflab = new QLabel ("Reference Station:", generalTab);
        QLabel *ref = new QLabel (rfnm, generalTab);
        ref->setFrameStyle (QFrame::Panel | QFrame::Sunken);
        QWhatsThis::add (reflab, refText);
        QWhatsThis::add (ref, refText);
    }


    QString stnm (rec.header.name);
    QLabel *namelab = new QLabel ("Station Name:", generalTab);
    name = new QLineEdit (stnm, generalTab);
    name->setFocus ();
    QWhatsThis::add (namelab, nameText);
    QWhatsThis::add (name, nameText);


    QString lats = QString (fixpos (rec.header.latitude, &deg, &min, &sec, 
        &hem, "lat", tideEditor->pos_format));
    QLabel *latlab = new QLabel ("Latitude:", generalTab);
    lat = new QLineEdit (lats, generalTab);
    QWhatsThis::add (latlab, latText);
    QWhatsThis::add (lat, latText);


    QString lons = QString (fixpos (rec.header.longitude, &deg, &min, &sec, 
        &hem, "lon", tideEditor->pos_format));
    QLabel *lonlab = new QLabel ("Longitude:", generalTab);
    lon = new QLineEdit (lons, generalTab);
    QWhatsThis::add (lonlab, lonText);
    QWhatsThis::add (lon, lonText);


    QString src = QString (rec.source);
    QLabel *sourcelab = new QLabel ("Source:", generalTab);
    source = new QLineEdit (src, generalTab);
    QWhatsThis::add (sourcelab, sourceText);
    QWhatsThis::add (source, sourceText);


    QLabel *tzfiles = new QLabel ("Time Zone:", generalTab);
    tzfiles->setMaximumHeight (tzfiles->sizeHint().height() * 2);
    tzfiles->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    tzfile = new QComboBox (FALSE, generalTab);
    QWhatsThis::add (tzfile, tzfileText);
    QWhatsThis::add (tzfiles, tzfileText);

    for (i = 0 ; i < db.tzfiles ; i++) 
    {
	QString str = QString (get_tzfile (i));
        tzfile->insertItem (str);
    }
    tzfile->setCurrentItem (rec.header.tzfile);


    QLabel *countrys = new QLabel ("Country:", generalTab);
    countrys->setMaximumHeight (countrys->sizeHint().height() * 2);
    countrys->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    country = new QComboBox (FALSE, generalTab);
    QWhatsThis::add (country, countryText);
    QWhatsThis::add (countrys, countryText);

    for (i = 0 ; i < db.countries ; i++) 
    {
	QString str = QString (get_country (i));
        country->insertItem (str);
    }
    country->setCurrentItem (rec.country);


    QLabel *pedigrees = new QLabel ("Pedigree:", generalTab);
    pedigrees->setMaximumHeight (pedigrees->sizeHint().height() * 2);
    pedigrees->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    pedigree = new QComboBox (FALSE, generalTab);
    QWhatsThis::add (pedigree, pedigreeText);
    QWhatsThis::add (pedigrees, pedigreeText);

    for (i = 0 ; i < db.pedigree_types ; i++) 
    {
	QString str = QString (get_pedigree (i));
        pedigree->insertItem (str);
    }
    pedigree->setCurrentItem (rec.pedigree);


    QLabel *restrictions = new QLabel ("Restriction:", generalTab);
    restrictions->setMaximumHeight (restrictions->sizeHint().height() * 2);
    restrictions->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    restriction = new QComboBox (FALSE, generalTab);
    QWhatsThis::add (restriction, restrictionText);
    QWhatsThis::add (restrictions, restrictionText);

    for (i = 0 ; i < db.restriction_types ; i++) 
    {
	QString str = QString (get_restriction (i));
        restriction->insertItem (str);
    }
    restriction->setCurrentItem (rec.restriction);


    QLabel *commentslab = new QLabel ("Comments:", generalTab);
    comments = new QTextEdit (generalTab);
    comments->setTextFormat (Qt::PlainText);
    QString cmt = QString (rec.comments);
    comments->setText (cmt);
    QWhatsThis::add (commentslab, commentsText);
    QWhatsThis::add (comments, commentsText);

    (void) QWhatsThis::whatsThisButton (generalTab);

    tideDialog->addTab (generalTab, "General");
}



//  Information that is unique to Reference stations.

void
TideDialog::constituentsTab ()
{
    int i;


    //  Don't display this if this is a Subordinate record.

    if (rec.header.record_type == 2) return;


    QVBox *constituentsTab = new QVBox (this);
    constituentsTab->setMargin (5);
    constituentsTab->setSpacing (5);

    QLabel *unitss = new QLabel ("Units:", constituentsTab);
    unitss->setMaximumHeight (unitss->sizeHint().height() * 2);
    unitss->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    units = new QComboBox (FALSE, constituentsTab);
    QWhatsThis::add (units, unitsText);
    QWhatsThis::add (unitss, unitsText);


    for (i = 0 ; i < db.level_unit_types ; i++) 
    {
	QString str = QString (get_level_units (i));
        units->insertItem (str);
    }
    units->setCurrentItem (rec.units);


    QLabel *datums = new QLabel ("Datum:", constituentsTab);
    datums->setMaximumHeight (datums->sizeHint().height() * 2);
    datums->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    datum = new QComboBox (FALSE, constituentsTab);
    QWhatsThis::add (datum, datumText);
    QWhatsThis::add (datums, datumText);


    for (i = 0 ; i < db.datum_types ; i++) 
    {
	QString str = QString (get_datum (i));
        datum->insertItem (str);
    }
    datum->setCurrentItem (rec.datum);


    QString datum_offsets = QString ("%1").arg (rec.datum_offset, 0, 'f', 4);
    QLabel *datum_offsetlab = new QLabel ("Datum Offset:", constituentsTab);
    datum_offset = new QLineEdit (datum_offsets, constituentsTab);
    QWhatsThis::add (datum_offsetlab, datum_offsetText);
    QWhatsThis::add (datum_offset, datum_offsetText);


    QString zone_offsets = QString ("%1").arg (rec.zone_offset);
    QLabel *zone_offsetlab = new QLabel ("Zone Offset:", constituentsTab);
    zone_offset = new QLineEdit (zone_offsets, constituentsTab);
    QWhatsThis::add (zone_offsetlab, zone_offsetText);
    QWhatsThis::add (zone_offset, zone_offsetText);


    QString expiration_dates = QString ("%1").arg (rec.expiration_date);
    if (expiration_dates == "0") expiration_dates = "";
    QLabel *expiration_datelab = 
        new QLabel ("Expiration Date:", constituentsTab);
    expiration_date = new QLineEdit (expiration_dates, constituentsTab);
    QWhatsThis::add (expiration_datelab, expiration_dateText);
    QWhatsThis::add (expiration_date, expiration_dateText);


    QString monthss = QString ("%1").arg (rec.months_on_station);
    if (monthss == "0") monthss = "";
    QLabel *monthslab = new QLabel ("Months On Station:", constituentsTab);
    months = new QLineEdit (monthss, constituentsTab);
    QWhatsThis::add (monthslab, monthsText);
    QWhatsThis::add (months, monthsText);


    QString ldoss = QString ("%1").arg (rec.last_date_on_station);
    if (ldoss == "0") ldoss = "";
    QLabel *ldoslab = new QLabel ("Last Date On Station:", constituentsTab);
    ldos = new QLineEdit (ldoss, constituentsTab);
    QWhatsThis::add (ldoslab, ldosText);
    QWhatsThis::add (ldos, ldosText);


    (void) new QLabel ("Constituents:", constituentsTab);

    constituents = new QTable (db.constituents, 2, constituentsTab);
    QWhatsThis::add (constituents, constituentsText);

    QHeader *hh = constituents->horizontalHeader ();
    hh->setLabel (0, QObject::tr ("Amplitude"), -1);
    hh->setLabel (1, QObject::tr ("Epoch"), -1);


    QHeader *vh = constituents->verticalHeader ();
    vh->setOrientation (Qt::Vertical);


    inf_rec = rec;
    infer_constituents (&inf_rec);

    QPixmap inferIcon = QPixmap (inferred);

    QMimeSourceFactory::defaultFactory ()->setPixmap ("inferred", inferIcon);


    for (i = 0 ; i < db.constituents ; i++)
    {
        QString cnstnm (cnst[i].name);
        vh->setLabel (i, cnstnm, -1);
        int j = cnst[i].num;

        QString amplitudes, epochs;
        if (rec.amplitude[j] == 0.0 && rec.epoch[j] == 0.0)
        {
            if (inf_rec.amplitude[j] == 0.0 && inf_rec.epoch[j] == 0.0)
            {
                amplitudes = QString ("");
                epochs = QString ("");
            }
            else
            {
                //  Inferred constituents.

                amplitudes = QString ("%1").arg (inf_rec.amplitude[j], 0, 
                    'f', 4);
                epochs = QString ("%1").arg (inf_rec.epoch[j], 0, 'f', 2);

                constituents->setPixmap (i, 0, inferIcon);
                constituents->setPixmap (i, 1, inferIcon);
            }
        }
        else
        {
            inf_rec.amplitude[j] = 0.0;
            inf_rec.epoch[j] = 0.0;


            amplitudes = QString ("%1").arg (rec.amplitude[j], 0, 'f', 4);
            epochs = QString ("%1").arg (rec.epoch[j], 0, 'f', 2);
        }

        constituents->setText (i, 0, amplitudes);
        constituents->setText (i, 1, epochs);


        //  Make sure that we can detect any change in the constituents 
        //  so that we only replace those that have changed.  This way we don't
        //  save inferred constituents unless they have been modified.

        sscanf (constituents->text (i, 0), "%f", &inf_rec.amplitude[j]);
        sscanf (constituents->text (i, 1), "%f", &inf_rec.epoch[j]);
    }


    (void) QWhatsThis::whatsThisButton (constituentsTab);

    tideDialog->addTab (constituentsTab, "Constituents");


    //  Check for subordinate stations so we can build a list tab if
    //  there are any.

    if (rec.header.record_number >= 0 && rec.header.reference_station == -1)
    {
        TIDE_RECORD temp_rec;
        int i, subs = 0;
        
        QVBox *subTab = new QVBox (this);
        subTab->setMargin (5);
        subTab->setSpacing (5);

        subList = new QListBox (subTab);
        QWhatsThis::add (subList, subListText);
        subList->setSelectionMode (QListBox::Extended);


        if (sub_station) free (sub_station);


        for (i = 0 ; i < db.number_of_records ; i++)
        {
            get_partial_tide_record (i, &temp_rec.header);

            if (temp_rec.header.reference_station == 
                rec.header.record_number)
            {
                double deg, min, sec;
                char hem;

                if ((sub_station = (int *) realloc (sub_station, (subs + 1) * 
                    sizeof (int))) == NULL)
                {
                    perror ("Allocating sub station memory");
                    exit (-1);
                }

                QString lats = QString (fixpos (temp_rec.header.latitude, 
                    &deg, &min, &sec, &hem, "lat", tideEditor->pos_format));
                QString lons = QString (fixpos (temp_rec.header.longitude, 
                    &deg, &min, &sec, &hem, "lon", tideEditor->pos_format));

                QString line = QString ("%1, %2   : %3")
                    .arg (lats, 13)
                    .arg (lons, 14)
                    .arg (temp_rec.header.name);

                subList->insertItem (line, -1);

                sub_station[subs] = temp_rec.header.record_number;
                subs++;
            }
        }


        //  If there were any, build the tab.

        if (subs)
        {
            QPushButton *changeButton = 
                new QPushButton ("Change Reference Station", subTab, 0);
            QWhatsThis::add (changeButton, subListText);

            connect (changeButton, SIGNAL (clicked ()), this, 
                SLOT (slotChangeRefStation ()));


            (void) QWhatsThis::whatsThisButton (subTab);

            tideDialog->addTab (subTab, "Subordinate Stations");
        }
        else
        {
            delete (subList);
        }
    }
}



//  Information that is unique to Subordinate stations.

void
TideDialog::offsetsTab ()
{
    int i;


    //  If this is a Reference station, don't display this tab.

    if (rec.header.record_type == 1) return;

    QVBox *offsetsTab = new QVBox (this);
    offsetsTab->setMargin (5);
    offsetsTab->setSpacing (5);


    QLabel *level_unitss = new QLabel ("Level Units:", offsetsTab);
    level_unitss->setMaximumHeight (level_unitss->sizeHint().height() * 2);
    level_unitss->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    level_units = new QComboBox (FALSE, offsetsTab);
    QWhatsThis::add (level_units, level_unitsText);
    QWhatsThis::add (level_unitss, level_unitsText);


    for (i = 0 ; i < db.level_unit_types ; i++) 
    {
	QString str = QString (get_level_units (i));
        level_units->insertItem (str);
    }
    level_units->setCurrentItem (rec.level_units);



    QLabel *direction_unitss = new QLabel ("Direction Units:", offsetsTab);
    direction_unitss->setMaximumHeight (direction_unitss->sizeHint().height() *
        2);
    direction_unitss->setFrameStyle (QFrame::Panel | QFrame::Sunken);
    direction_units = new QComboBox (FALSE, offsetsTab);
    QWhatsThis::add (direction_units, direction_unitsText);
    QWhatsThis::add (direction_unitss, direction_unitsText);


    for (i = 0 ; i < db.dir_unit_types ; i++) 
    {
	QString str = QString (get_dir_units (i));
        direction_units->insertItem (str);
    }
    direction_units->setCurrentItem (rec.direction_units);


    QString min_time_adds = QString ("%1").arg (rec.min_time_add);
    if (min_time_adds == "0") min_time_adds = "";
    QLabel *min_time_addlab = new QLabel ("Minimum Time Add:", offsetsTab);
    min_time_add = new QLineEdit (min_time_adds, offsetsTab);
    QWhatsThis::add (min_time_addlab, min_time_addText);
    QWhatsThis::add (min_time_add, min_time_addText);


    QString min_level_adds = QString ("%1").arg (rec.min_level_add, 0, 'f', 
        2);
    if (min_level_adds == "0.00") min_level_adds = "";
    QLabel *min_level_addlab = new QLabel ("Minimum Level Add:", offsetsTab);
    min_level_add = new QLineEdit (min_level_adds, offsetsTab);
    QWhatsThis::add (min_level_addlab, min_level_addText);
    QWhatsThis::add (min_level_add, min_level_addText);


    QString min_level_mults = QString ("%1").arg (rec.min_level_multiply, 0, 
        'f', 2);
    if (min_level_mults == "0.00") min_level_mults = "";
    QLabel *min_level_multlab = 
        new QLabel ("Minimum Level Multiply:", offsetsTab);
    min_level_mult = new QLineEdit (min_level_mults, offsetsTab);
    QWhatsThis::add (min_level_multlab, min_level_multText);
    QWhatsThis::add (min_level_mult, min_level_multText);


    QString min_avg_levels = QString ("%1").arg (rec.min_avg_level, 0, 'f', 
        2);
    if (min_avg_levels == "0.00") min_avg_levels = "";
    QLabel *min_avg_levellab =  
        new QLabel ("Minimum Average Level:", offsetsTab);
    min_avg_level = new QLineEdit (min_avg_levels, offsetsTab);
    QWhatsThis::add (min_avg_levellab, min_avg_levelText);
    QWhatsThis::add (min_avg_level, min_avg_levelText);


    QString min_directions = QString ("%1").arg (rec.min_direction);
    if (min_directions == "361") min_directions = "";
    QLabel *min_directionlab = new QLabel ("Minimum Direction:", offsetsTab);
    min_direction = new QLineEdit (min_directions, offsetsTab);
    QWhatsThis::add (min_directionlab, min_directionText);
    QWhatsThis::add (min_direction, min_directionText);


    QString max_time_adds = QString ("%1").arg (rec.max_time_add);
    if (max_time_adds == "0") max_time_adds = "";
    QLabel *max_time_addlab = new QLabel ("Maximum Time Add:", offsetsTab);
    max_time_add = new QLineEdit (max_time_adds, offsetsTab);
    QWhatsThis::add (max_time_addlab, max_time_addText);
    QWhatsThis::add (max_time_add, max_time_addText);


    QString max_level_adds = QString ("%1").arg (rec.max_level_add, 0, 'f', 
        2);
    if (max_level_adds == "0.00") max_level_adds = "";
    QLabel *max_level_addlab = new QLabel ("Maximum Level Add:", offsetsTab);
    max_level_add = new QLineEdit (max_level_adds, offsetsTab);
    QWhatsThis::add (max_level_addlab, max_level_addText);
    QWhatsThis::add (max_level_add, max_level_addText);


    QString max_level_mults = QString ("%1").arg (rec.max_level_multiply, 0,
        'f', 2);
    if (max_level_mults == "0.00") max_level_mults = "";
    QLabel *max_level_multlab = 
        new QLabel ("Maximum Level Multiply:", offsetsTab);
    max_level_mult = new QLineEdit (max_level_mults, offsetsTab);
    QWhatsThis::add (max_level_multlab, max_level_multText);
    QWhatsThis::add (max_level_mult, max_level_multText);


    QString max_avg_levels = QString ("%1").arg (rec.max_avg_level, 0, 'f', 
        2);
    if (max_avg_levels == "0.00") max_avg_levels = "";
    QLabel *max_avg_levellab = 
        new QLabel ("Maximum Average Level:", offsetsTab);
    max_avg_level = new QLineEdit (max_avg_levels, offsetsTab);
    QWhatsThis::add (max_avg_levellab, max_avg_levelText);
    QWhatsThis::add (max_avg_level, max_avg_levelText);


    QString max_directions = QString ("%1").arg (rec.max_direction);
    if (max_directions == "361") max_directions = "";
    QLabel *max_directionlab = new QLabel ("Maximum Direction:", offsetsTab);
    max_direction = new QLineEdit (max_directions, offsetsTab);
    QWhatsThis::add (max_directionlab, max_directionText);
    QWhatsThis::add (max_direction, max_directionText);


    /* DWF updated per slack offsets fix 2003-03-18 */
    QString flood_beginss = QString ("%1").arg (rec.flood_begins);
    if (flood_beginss == "2560") flood_beginss = "";
    QLabel *flood_beginslab = new QLabel ("Flood Begins:", offsetsTab);
    flood_begins = new QLineEdit (flood_beginss, offsetsTab);
    QWhatsThis::add (flood_beginslab, flood_beginsText);
    QWhatsThis::add (flood_begins, flood_beginsText);


    /* DWF updated per slack offsets fix 2003-03-18 */
    QString ebb_beginss = QString ("%1").arg (rec.ebb_begins);
    if (ebb_beginss == "2560") ebb_beginss = "";
    QLabel *ebb_beginslab = new QLabel ("Ebb Begins:", offsetsTab);
    ebb_begins = new QLineEdit (ebb_beginss, offsetsTab);
    QWhatsThis::add (ebb_beginslab, ebb_beginsText);
    QWhatsThis::add (ebb_begins, ebb_beginsText);

    (void) QWhatsThis::whatsThisButton (offsetsTab);


    tideDialog->addTab (offsetsTab, "Offsets");
}



//  Update station data in tide DB.

void
TideDialog::slotUpdateStation ()
{
    int i;
    double inlat, inlon;
    char string[50];


    // Common data.

    strcpy (rec.header.name, name->text ());
    strcpy (rec.source, source->text ());

    strcpy (string, lat->text ());
    posfix (string, &inlat, "lat");
    strcpy (string, lon->text ());
    posfix (string, &inlon, "lon");

    if (inlat != rec.header.latitude || inlon != rec.header.longitude)
    {
        rec.header.latitude = inlat;
        rec.header.longitude = inlon;
        tideEditor->setRedrawZeroLevel ();
    }


    rec.header.tzfile = tzfile->currentItem ();
    rec.country = country->currentItem ();
    rec.pedigree = pedigree->currentItem ();
    rec.restriction = restriction->currentItem ();
    strcpy (rec.comments, comments->text ());


    //  Reference station data.

    if (rec.header.record_type == 1)
    {
        rec.units = units->currentItem ();
        rec.datum = datum->currentItem ();
        sscanf (datum_offset->text (), "%f", &rec.datum_offset);
        sscanf (zone_offset->text (), "%d", &rec.zone_offset);
        if (sscanf (expiration_date->text (), "%d", &rec.expiration_date) <= 0)
            rec.expiration_date = 0;
        // DWF: fixed warning: int format, different type arg
        {
          int tempm;
          if (sscanf (months->text (), "%d", &tempm) <= 0)
              rec.months_on_station = (NV_U_BYTE)0;
          else
	      rec.months_on_station = (NV_U_BYTE)tempm;
        }
        if (sscanf (ldos->text (), "%d", &rec.last_date_on_station) <= 0)
            rec.last_date_on_station = 0;

        for (i = 0 ; i < db.constituents ; i++)
        {
            float amp, epo;
            if (sscanf (constituents->text (i, 0), "%f", &amp) == EOF) 
                amp = 0.0;
            if (sscanf (constituents->text (i, 1), "%f", &epo) == EOF) 
                epo = 0.0;

            int j = cnst[i].num;

            //  Check for a change to constituents.  If there is no
            //  change, don't replace the constituents.

            if (amp != inf_rec.amplitude[j] || epo != inf_rec.epoch[j])
            {
                rec.amplitude[j] = amp;
                rec.epoch[j] = epo;
            }
        }
    }


    //  Subordinate station data.

    else
    {
        rec.level_units = level_units->currentItem ();
        rec.direction_units = direction_units->currentItem ();
        if (sscanf (min_time_add->text (), "%d", &rec.min_time_add) <= 0)
            rec.min_time_add = 0;
        if (sscanf (min_level_add->text (), "%f", &rec.min_level_add) <= 0)
            rec.min_level_add = 0;
        if (sscanf (min_level_mult->text (), "%f", &rec.min_level_multiply) <=
            0) rec.min_level_multiply = 0;
        if (sscanf (min_avg_level->text (), "%f", &rec.min_avg_level) <= 0)
            rec.min_avg_level = 0;
        if (sscanf (min_direction->text (), "%d", &rec.min_direction) <= 0)
            rec.min_direction = 361;
        if (sscanf (max_time_add->text (), "%d", &rec.max_time_add) <= 0)
            rec.max_time_add = 0;
        if (sscanf (max_level_add->text (), "%f", &rec.max_level_add) <= 0)
            rec.max_level_add = 0;
        if (sscanf (max_level_mult->text (), "%f", &rec.max_level_multiply) <=
            0) rec.max_level_multiply = 0;
        if (sscanf (max_avg_level->text (), "%f", &rec.max_avg_level) <= 0)
            rec.max_avg_level = 0;
        if (sscanf (max_direction->text (), "%d", &rec.max_direction) <= 0)
            rec.max_direction = 361;
        /* DWF updated per slack offsets fix 2003-03-18 */
        if (sscanf (flood_begins->text (), "%d", &rec.flood_begins) <= 0)
            rec.flood_begins = NULLSLACKOFFSET;
        /* DWF updated per slack offsets fix 2003-03-18 */
        if (sscanf (ebb_begins->text (), "%d", &rec.ebb_begins) <= 0)
            rec.ebb_begins = NULLSLACKOFFSET;
    }


    //  Clear memory.

    slotCancelStation ();


    //  Update the record in the database, add it if the header.record_number
    //  is less than zero.

    if (rec.header.record_number >= 0)
    {
        update_tide_record (rec.header.record_number, &rec);
    }
    else
    {
        add_tide_record (&rec, &db);

        tideEditor->setRedrawZeroLevel ();
    }


    tideEditor->redrawMap ();
}



//  Cancel any update or write.

void
TideDialog::slotCancelStation ()
{
    if (rec.header.record_type == 1)
    {
        delete (units);
        delete (datum);
        delete (datum_offset);
        delete (zone_offset);
        delete (expiration_date);
        delete (months);
        delete (ldos);
        delete (constituents);
    }
    else
    {
        delete (level_units);
        delete (direction_units);
        delete (min_time_add);
        delete (min_level_add);
        delete (min_level_mult);
        delete (min_avg_level);
        delete (min_direction);
        delete (max_time_add);
        delete (max_level_add);
        delete (max_level_mult);
        delete (max_avg_level);
        delete (max_direction);
        delete (flood_begins);
        delete (ebb_begins);
    }


    delete (name);
    delete (source);
    delete (lat);
    delete (lon);
    delete (tzfile);
    delete (country);
    delete (pedigree);
    delete (restriction);
    delete (comments);
    delete (tideDialog);
}



//  Deleting a station.

void 
TideDialog::slotDeleteStation ()
{
    //  Clear memory.

    slotCancelStation ();


    QMessageBox mb ("Delete Tide Station",
        "Do you really want to delete this station?\n"
        "There is no way to recover once it is deleted!\n\n",
        QMessageBox::Warning,
        QMessageBox::Yes | QMessageBox::Escape,
        QMessageBox::No  | QMessageBox::Default,
        0);

    if (mb.exec() == QMessageBox::No) return;


    //  Check for subordinate stations.

    if (rec.header.record_type == 1 && rec.header.reference_station == -1)
    {
        TIDE_RECORD temp_rec;
        int i, subs = 0;
        
        for (i = 0 ; i < db.number_of_records ; i++)
        {
            get_partial_tide_record (i, &temp_rec.header);

            if (temp_rec.header.reference_station == 
                rec.header.record_number) subs++;
        }

        if (subs)
        {
            QString msg = 
                QString ("This station has %1 subordinate stations!\n"
                "Deleting it will delete all of the subordinate stations.\n"
                "Are you sure you want to delete it?\n\n")
                .arg (subs);

            QMessageBox mb ("Delete Reference Tide Station", msg,
                QMessageBox::Critical,
                QMessageBox::Yes | QMessageBox::Escape,
                QMessageBox::No  | QMessageBox::Default,
                0);

            if (mb.exec() == QMessageBox::No) return;
        }
    }


    delete_tide_record (rec.header.record_number, &db);


    tideEditor->setRedrawZeroLevel ();

    tideEditor->redrawMap ();
}



//  Print station information.

void
TideDialog::slotPrintStation ()
{
    QPrinter *printer = new QPrinter (QPrinter::ScreenResolution);


    printer->setFullPage (TRUE);
    printer->setPageSize (QPrinter::Letter);

    if (printer->setup (tideDialog))
      {
        QTextEdit *e = new QTextEdit (tideDialog, "editor");
        QString *s = new QString ();
        char string[256];

        sprintf (string, "\nStation Name: %s\n", rec.header.name);
        s->append (string);
        sprintf (string, "Latitude: %+.6f\n", rec.header.latitude);
        s->append (string);
        sprintf (string, "Longitude: %+.6f\n", rec.header.longitude);
        s->append (string);
        sprintf (string, "Source: %s\n", rec.source);
        s->append (string);
        sprintf (string, "Time Zone: %s\n", get_tzfile (rec.header.tzfile));
        s->append (string);
        sprintf (string, "Country: %s\n", get_country (rec.country));
        s->append (string);
        sprintf (string, "Pedigree: %s\n", get_pedigree (rec.pedigree));
        s->append (string);
        sprintf (string, "Restriction: %s\n", 
                 get_restriction (rec.restriction));
        s->append (string);
        sprintf (string, "Comments: %s\n", rec.comments);
        s->append (string);


        if (rec.header.record_type == 1)
          {
            int i;

            sprintf (string, "Units: %s\n", get_level_units (rec.units));
            s->append (string);
            sprintf (string, "Datum: %s\n", get_datum (rec.datum));
            s->append (string);
            sprintf (string, "Datum Offset: %f\n", rec.datum_offset);
            s->append (string);
            sprintf (string, "Zone Offset: %d\n", rec.zone_offset);
            s->append (string);
            if (rec.expiration_date)
              {
                sprintf (string, "Expiration Date: %d\n", rec.expiration_date);
                s->append (string);
              }
            if (rec.months_on_station)
              {
                sprintf (string, "Months On Station: %d\n", 
                         rec.months_on_station);
                s->append (string);
              }
            if (rec.last_date_on_station)
              {
                sprintf (string, "Last Date On Station: %d\n", 
                         rec.last_date_on_station);
                s->append (string);
              }


            sprintf (string, "\n\nConstituents:\n\n");
            s->append (string);

            for (i = 0 ; i < db.constituents ; i++)
              {
                int j = cnst[i].num;
                if (rec.amplitude[j] == 0.0 && rec.epoch[j] == 0.0)
                  {
                    if (inf_rec.amplitude[j] != 0.0 || inf_rec.epoch[j] != 0.0)
                      {
                        //  Inferred constituents->

                        sprintf (string, 
                            "%s: \tamplitude: %f \tepoch: %f \t(inferred)\n",
                            cnst[i].name, inf_rec.amplitude[j], 
                            inf_rec.epoch[j]);
                        s->append (string);
                      }
                  }
                else
                  {
                    sprintf (string, "%s: \tamplitude: %f \tepoch: %f\n",
                             cnst[i].name, rec.amplitude[j], rec.epoch[j]);
                    s->append (string);
                  }
              }


            TIDE_RECORD temp_rec;
            int subs = 0;
        
            for (i = 0 ; i < db.number_of_records ; i++)
              {
                get_partial_tide_record (i, &temp_rec.header);

                if (temp_rec.header.reference_station == 
                    rec.header.record_number)
                  {
                    if (!subs)
                      {
                        sprintf (string, "\n\nSubordinate Stations:\n\n");
                        s->append (string);
                      }

                    sprintf (string, "%s\n", temp_rec.header.name);
                    s->append (string);
                    subs++;
                  }                    
              }
          }
        else
          {
            sprintf (string, "\nReference Station: %s\n", 
                     get_station (rec.header.reference_station));
            s->prepend (string);
            sprintf (string, "Level Units: %s\n", 
                     get_level_units (rec.level_units));
            s->append (string);
            sprintf (string, "Direction Units: %s\n", 
                     get_dir_units (rec.direction_units));
            s->append (string);
            if (rec.min_time_add)
              {
                sprintf (string, "Minimum Time Add: %d\n", rec.min_time_add);
                s->append (string);
              }
            if (rec.min_level_add != 0.00)
              {
                sprintf (string, "Minimum Level Add: %f\n", rec.min_level_add);
                s->append (string);
              }
            if (rec.min_level_multiply != 0.00)
              {
                sprintf (string, "Minimum Level Multiply: %f\n", 
                         rec.min_level_multiply);
                s->append (string);
              }
            if (rec.min_avg_level != 0.00)
              {
                sprintf (string, "Minimum Average Level: %f\n", 
                         rec.min_avg_level);
                s->append (string);
              }
            if (rec.min_direction != 361)
              {
                sprintf (string, "Minimum Direction: %d\n", rec.min_direction);
                s->append (string);
              }
            if (rec.max_time_add)
              {
                sprintf (string, "Maximum Time Add: %d\n", rec.max_time_add);
                s->append (string);
              }
            if (rec.max_level_add != 0.00)
              {
                sprintf (string, "Maximum Level Add: %f\n", rec.max_level_add);
                s->append (string);
              }
            if (rec.max_level_multiply != 0.00)
              {
                sprintf (string, "Maximum Level Multiply: %f\n", 
                         rec.max_level_multiply);
                s->append (string);
              }
            if (rec.max_avg_level != 0.00)
              {
                sprintf (string, "Maximum Average Level: %f\n", 
                         rec.max_avg_level);
                s->append (string);
              }
            if (rec.max_direction != 361)
              {
                sprintf (string, "Maximum Direction: %d\n", rec.max_direction);
                s->append (string);
              }
            if (rec.flood_begins != 2560)
              {
                sprintf (string, "Flood Begins: %d\n", rec.flood_begins);
                s->append (string);
              }
            if (rec.ebb_begins != 2560)
              {
                sprintf (string, "Ebb Begins: %d\n", rec.ebb_begins);
                s->append (string);
              }
          }

        e->setText (*s);


        QPainter p;
        if (!p.begin (printer)) return;


        QPaintDeviceMetrics metrics (p.device ());
        int dpiy = metrics.logicalDpiY ();
        int margin = (int) ((2 / 2.54) * dpiy);                 // 2 cm margins
        QRect body (margin, margin, metrics.width() - 2 * margin, 
                    metrics.height () - 2 * margin);
        QSimpleRichText richText (
            QStyleSheet::convertFromPlainText (e->text ()), QFont (),
            e->context (), e->styleSheet (), e->mimeSourceFactory (),
            body.height());
        richText.setWidth (&p, body.width ());
        QRect view (body);
        int page = 1;
        do 
          {
            richText.draw (&p, body.left (), body.top (), view, 
                           colorGroup ());
            view.moveBy (0, body.height ());
            p.translate (0, -body.height ());
            p.drawText (view.right () - 
                        p.fontMetrics ().width (QString::number (page)),
                        view.bottom () + 
                        p.fontMetrics ().ascent () + 5, 
                        QString::number (page));
            if (view.top () >= richText.height ()) break;
            printer->newPage ();
            page++;
          } while (TRUE);
      } 
}



//  Change the Reference station ID for one or more Subordinate stations.

void 
TideDialog::slotChangeRefStation ()
{
    TIDE_RECORD temp_rec;
    int count = 0;
    QStringList strlist;


    //  Build a list of displayed stations (not all stations) that have
    //  a header.reference_station value of -1 (true Reference stations not
    //  Subordinate stations with a set of constituents).

    for (int i = 0 ; i < tideEditor->number_of_records ; i++)
    {
        get_partial_tide_record (tideEditor->station_list[i].recnum, 
            &temp_rec.header);
        if (temp_rec.header.reference_station == -1)
        {
            strlist += tideEditor->station_list[i].name;
            count++;
        }
    }


    //  If there were any in the current displayed area pop up the input
    //  dialog and get the selected station.

    if (count)
    {
        bool ok = FALSE;
        QString res = QInputDialog::getItem (tr ("Reference Stations"),
            tr ("Please select the new reference station"), strlist, 0, FALSE,
            &ok, (QWidget *) this);

        if (ok)
        {
            char station[NAME_LENGTH];
            strcpy (station, res);

            int num = find_station (station);
            for (int i = 0 ; i < subList->count () ; i++)
            {
                if (subList->isSelected (i))
                {
                    read_tide_record (sub_station[i], &temp_rec);
                    temp_rec.header.reference_station = num;
                    update_tide_record (sub_station[i], &temp_rec);
                }
            }
        }
    }
}
