/*
 *   meiouart.c -- MEIO device dependant code for UART connections
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <port_types.h> 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>

#include <meio_api.h>          /* include general MEIO info */
#include "meiomacs.h"          /* include vendor driver types and macros */
#include <meiogtwy.h>
ResourceSRef(CommTx1_UartOut1    );
ResourceSRef(UartIn1_CommRx1     );

#define UARTITCB  "UARTITCB"
#define MEIOUART  "MEIOUART_ITCB"
#define UARTTASK  "UART000"
#define UARTPARMS "UCY_PARM"
#define MIN_UART_CPF   1        /* minimum number chars per frame */

static long   UARTchars_per_second = 300;
static int    UARTownercount = 0;
static HMTASK UARThmTask = 0;
static ULONG  ulUARTparmaddr = 0;
static ULONG  GetTaskCPF(HMTASK hTask, ULONG FAR *pulCPF);

/********************************************************/
/* GetTaskCPF() - get the Cycles per Frame for a task.  */
/********************************************************/
ULONG GetTaskCPF(HMTASK hTask, ULONG FAR *pulCPF)
{
  ULONG    ulRC;
  static struct
  {
    RTASKINFO tib;
    ULONG     junkdata[30];    /* enough room for the extra crap returned */
  } TaskData;
  USHORT   usSize = sizeof(TaskData);
  
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meiouart::GetTaskCPF entry hTask %p\n",hTask);
  ulRC=dspQueryTaskInfo(hTask,&usSize,&TaskData);
  if (ulRC == DSP_NOERROR) {
    *pulCPF = TaskData.tib.TSKINFO_ulCPF;
    if (0xffffffffL==*pulCPF) *pulCPF = 0;  /* no cycles */
  }
  MW_SYSLOG_3(TRACE_MEIO_SPECIFIC,"meiouart::GetTaskCPF exit ulCPF %lx ulRC %lx\n",*pulCPF,ulRC);
  return (ulRC);
}

ULONG UpdateUARTcpf( PMEIO_DSP pD)
{
  ULONG ulRC = 0;
  long CurrentCPF, NewCPF;
  long newUARTcpf;
  static struct
  {
    short FixedCycles;      /* fixed number of cycles in UART task */
    short VarCycles;        /* number of cycles per loop */
    short CharsPerFrame;    /* current number of characters per frame */
  } UARTparm;
  
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meiouart::UpdateUARTcpf entry pD %p\n",pD);
  do {
    if (ulUARTparmaddr) {      /* do we have a valid address? */
      ulRC = dspMemTransfer(pD->hDsp,                /* from this dsp */
			    ulUARTparmaddr,          /* get from DSP Addr... */
			    &UARTparm,               /* put data here */
			    sizeof(UARTparm)/2,      /* number of WORDS */
			    DSP_MEMXFER_DATA_READ);
      if (ulRC!=DSP_NOERROR) break;
      
      /* new chars per frame = ceil(cps/fps) = cps/44100/32 */
      /*                       to get ceil function add 44099 before divide */
      newUARTcpf = ((UARTchars_per_second * 32L)+44099L)/44100L;
      
      /* max(newUARTcpf,MIN_UART_CPF) */
      if (newUARTcpf < MIN_UART_CPF)
	newUARTcpf = MIN_UART_CPF;
      
      /* only update task if change is required */
      if (UARTparm.CharsPerFrame != (short)newUARTcpf) {
	UARTparm.CharsPerFrame = (short)newUARTcpf;
	
	NewCPF = (newUARTcpf * UARTparm.VarCycles)+UARTparm.FixedCycles;
	
	/* now get current CPF */
	ulRC = GetTaskCPF(UARThmTask, (ULONG FAR *)&CurrentCPF);
	if (ulRC!=DSP_NOERROR) break;
	
	/* sequence is IMPORTANT! */
	if (NewCPF < CurrentCPF) {   /* are we reducing CPF? */
	  ulRC = dspMemTransfer(pD->hDsp,                /* from this dsp */
				ulUARTparmaddr+4,        /* get from DSP Addr... */
				&UARTparm.CharsPerFrame, /* put data here */
				1,                       /* number of WORDS */
				DSP_MEMXFER_DATA_WRITE);
	  if (ulRC!=DSP_NOERROR) break;
	  
	  ulRC = dspChangeCPF(UARThmTask,NewCPF-CurrentCPF);
	  
	} else {
	  ulRC = dspChangeCPF(UARThmTask,NewCPF-CurrentCPF);
	  if (ulRC!=DSP_NOERROR) break;
	  
	  ulRC = dspMemTransfer(pD->hDsp,                /* from this dsp */
				ulUARTparmaddr+4,        /* get from DSP Addr... */
				&UARTparm.CharsPerFrame, /* put data here */
				1,                       /* number of WORDS */
				DSP_MEMXFER_DATA_WRITE);
	} 
      } 
    } else {
      /* say that we can't support this call */
      return MEIO_NC_INVALID_ATTRIBUTE;
    }
  } while ( 0 ); /* do only once */
  /* return manager error as unique error code */
  
  if (ulRC) ulRC = 0x10000000 | (0x0fffffff & ulRC);
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meiouart::UpdateUARTcpf exit ulRC %lx\n",ulRC);
  return ulRC;
}

/*-----------------------------------------------------------------------
 *
 *  Setup UART Interface
 *
 *----------------------------------------------------------------------- */
ULONG SetupUART( PMEIO_DSP pD)
{
  ULONG ulRC=MEIO_NOERROR;
  
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meiouart::SetupUART entry pD %p\n",pD);
  
  if (UARTownercount > 0) {
    ulUARTparmaddr = 0;           /* in case setup fails */
    ulRC = dspConnectITCB(  0,        UARTITCB,
			    pD->hmTask,MEIOUART);
    
    if (ulRC==DSP_NOERROR)
      ulRC = dspNameToTaskHandle(pD->hMwaveMod,
				 UARTTASK,
				 &UARThmTask);
    
    if (ulRC==DSP_NOERROR)
      ulRC = dspLabelToAddress(UARThmTask, UARTPARMS,
			       &ulUARTparmaddr);
    
    if (ulRC)
      ulRC = 0x10000000 | (0x0fffffff & ulRC);
    
  }
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meiouart::SetupUART exit ulRC %lx\n",ulRC);
  return ulRC;
}
/*-----------------------------------------------------------------------
 *
 *  UART connection is fixed, report that it is always ON
 *  New methods for reserve, release, and CONNECTION_CHARACTERISTICS
 *
 *----------------------------------------------------------------------- */
static ULONG UARTconnection(RF_ARGS)
{
  PMEIO_DSP pD;
  ULONG ulRC = 0;
    
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meiouart::UARTconnection entry usMessage %x\n",usMessage);
  /* Connection State is considered always ON */
  if (usMessage==mQUERYmsg && lParam1 == MEIO_CONNECTION_STATE) {
    *((LPLONG)lParam2) = ON;
    return(MEIO_NOERROR);
  } else {
    /* Connection Characteristics are contained in MeioTAIOCharacteristics */
    if (usMessage==mQUERYmsg && lParam1 == MEIO_CONNECTION_CHARACTERISTICS) {
      *((LPLONG)lParam2) = UARTchars_per_second;
      return(MEIO_NOERROR);
    } else {
      /* Connection Characteristics are contained in MeioTAIOCharacteristics */
      if (usMessage==mUPDATEmsg && lParam1 == MEIO_CONNECTION_CHARACTERISTICS) {
	pD = pConnection->pMeioDsp;
	UARTchars_per_second = lParam2;
	return UpdateUARTcpf(pD);
      } else {
	if (usMessage==mRESERVEmsg && 0==UARTownercount++) {
	  pD = pConnection->pMeioDsp;
	  ulRC = SetupUART(pD);
	  
	} else {
	  if (usMessage==mRELEASEmsg && 1==UARTownercount--) {
	    pD = pConnection->pMeioDsp;
	    ulRC = dspDisconnectITCB(  0,        UARTITCB,
				       pD->hmTask,MEIOUART);
	    ulUARTparmaddr = 0;
	    UARThmTask = 0;
	  }
	}
      }
    }
  }
  
  (void)ulRC;   /* make compiler happy */
  
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meiouart::UARTconnection exit direct from generic_cf ulRC %lx\n",ulRC);
  if (ulRC) return ulRC;
  else return generic_cf(RF_ARGLIST);
  
}
ULONG FAR CommTx1_UartOut1     (RF_ARGS) { return UARTconnection(RF_ARGLIST); }
ULONG FAR UartIn1_CommRx1      (RF_ARGS) { return UARTconnection(RF_ARGLIST); }

