'use strict';

module.exports = function() {
  // KMAC test vectors from https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/KMAC_samples.pdf
  const vectors = [
    {
      // Sample #1 - KMAC128, no customization
      algorithm: 'KMAC128',
      key: Buffer.from([
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b,
        0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
        0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
      ]),
      data: Buffer.from([0x00, 0x01, 0x02, 0x03]),
      customization: undefined,
      length: 256,
      expected: Buffer.from([
        0xe5, 0x78, 0x0b, 0x0d, 0x3e, 0xa6, 0xf7, 0xd3, 0xa4, 0x29, 0xc5, 0x70,
        0x6a, 0xa4, 0x3a, 0x00, 0xfa, 0xdb, 0xd7, 0xd4, 0x96, 0x28, 0x83, 0x9e,
        0x31, 0x87, 0x24, 0x3f, 0x45, 0x6e, 0xe1, 0x4e,
      ]),
    },
    {
      // Sample #2 - KMAC128, with customization
      algorithm: 'KMAC128',
      key: Buffer.from([
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b,
        0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
        0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
      ]),
      data: Buffer.from([0x00, 0x01, 0x02, 0x03]),
      customization: Buffer.from('My Tagged Application'),
      length: 256,
      expected: Buffer.from([
        0x3b, 0x1f, 0xba, 0x96, 0x3c, 0xd8, 0xb0, 0xb5, 0x9e, 0x8c, 0x1a, 0x6d,
        0x71, 0x88, 0x8b, 0x71, 0x43, 0x65, 0x1a, 0xf8, 0xba, 0x0a, 0x70, 0x70,
        0xc0, 0x97, 0x9e, 0x28, 0x11, 0x32, 0x4a, 0xa5,
      ]),
    },
    {
      // Sample #3 - KMAC128, large data, with customization
      algorithm: 'KMAC128',
      key: Buffer.from([
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b,
        0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
        0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
      ]),
      data: Buffer.from(Array.from({ length: 200 }, (_, i) => i)), // 0x00-0xC7
      customization: Buffer.from('My Tagged Application'),
      length: 256,
      expected: Buffer.from([
        0x1f, 0x5b, 0x4e, 0x6c, 0xca, 0x02, 0x20, 0x9e, 0x0d, 0xcb, 0x5c, 0xa6,
        0x35, 0xb8, 0x9a, 0x15, 0xe2, 0x71, 0xec, 0xc7, 0x60, 0x07, 0x1d, 0xfd,
        0x80, 0x5f, 0xaa, 0x38, 0xf9, 0x72, 0x92, 0x30,
      ]),
    },
    {
      // Sample #4 - KMAC256, with customization, 512-bit output
      algorithm: 'KMAC256',
      key: Buffer.from([
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b,
        0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
        0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
      ]),
      data: Buffer.from([0x00, 0x01, 0x02, 0x03]),
      customization: Buffer.from('My Tagged Application'),
      length: 512,
      expected: Buffer.from([
        0x20, 0xc5, 0x70, 0xc3, 0x13, 0x46, 0xf7, 0x03, 0xc9, 0xac, 0x36, 0xc6,
        0x1c, 0x03, 0xcb, 0x64, 0xc3, 0x97, 0x0d, 0x0c, 0xfc, 0x78, 0x7e, 0x9b,
        0x79, 0x59, 0x9d, 0x27, 0x3a, 0x68, 0xd2, 0xf7, 0xf6, 0x9d, 0x4c, 0xc3,
        0xde, 0x9d, 0x10, 0x4a, 0x35, 0x16, 0x89, 0xf2, 0x7c, 0xf6, 0xf5, 0x95,
        0x1f, 0x01, 0x03, 0xf3, 0x3f, 0x4f, 0x24, 0x87, 0x10, 0x24, 0xd9, 0xc2,
        0x77, 0x73, 0xa8, 0xdd,
      ]),
    },
    {
      // Sample #5 - KMAC256, large data, no customization, 512-bit output
      algorithm: 'KMAC256',
      key: Buffer.from([
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b,
        0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
        0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
      ]),
      data: Buffer.from(Array.from({ length: 200 }, (_, i) => i)), // 0x00-0xC7
      customization: undefined,
      length: 512,
      expected: Buffer.from([
        0x75, 0x35, 0x8c, 0xf3, 0x9e, 0x41, 0x49, 0x4e, 0x94, 0x97, 0x07, 0x92,
        0x7c, 0xee, 0x0a, 0xf2, 0x0a, 0x3f, 0xf5, 0x53, 0x90, 0x4c, 0x86, 0xb0,
        0x8f, 0x21, 0xcc, 0x41, 0x4b, 0xcf, 0xd6, 0x91, 0x58, 0x9d, 0x27, 0xcf,
        0x5e, 0x15, 0x36, 0x9c, 0xbb, 0xff, 0x8b, 0x9a, 0x4c, 0x2e, 0xb1, 0x78,
        0x00, 0x85, 0x5d, 0x02, 0x35, 0xff, 0x63, 0x5d, 0xa8, 0x25, 0x33, 0xec,
        0x6b, 0x75, 0x9b, 0x69,
      ]),
    },
    {
      // Sample #6 - KMAC256, large data, with customization, 512-bit output
      algorithm: 'KMAC256',
      key: Buffer.from([
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b,
        0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
        0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
      ]),
      data: Buffer.from(Array.from({ length: 200 }, (_, i) => i)), // 0x00-0xC7
      customization: Buffer.from('My Tagged Application'),
      length: 512,
      expected: Buffer.from([
        0xb5, 0x86, 0x18, 0xf7, 0x1f, 0x92, 0xe1, 0xd5, 0x6c, 0x1b, 0x8c, 0x55,
        0xdd, 0xd7, 0xcd, 0x18, 0x8b, 0x97, 0xb4, 0xca, 0x4d, 0x99, 0x83, 0x1e,
        0xb2, 0x69, 0x9a, 0x83, 0x7d, 0xa2, 0xe4, 0xd9, 0x70, 0xfb, 0xac, 0xfd,
        0xe5, 0x00, 0x33, 0xae, 0xa5, 0x85, 0xf1, 0xa2, 0x70, 0x85, 0x10, 0xc3,
        0x2d, 0x07, 0x88, 0x08, 0x01, 0xbd, 0x18, 0x28, 0x98, 0xfe, 0x47, 0x68,
        0x76, 0xfc, 0x89, 0x65,
      ]),
    },
  ];

  return vectors;
};
