/*
 * Audacious: Cross platform multimedia player
 * Copyright (c) 2005-2009 Audacious Team
 *
 * Driver for Game_Music_Emu library. See details at:
 * http://www.slack.net/~ant/libs/
 */

#include <math.h>
#include <stdio.h>
#include <string.h>

extern "C" {
#include <libaudcore/audstrings.h>
#include <audacious/input.h>
#include <audacious/plugin.h>
}

#include "configure.h"
#include "Music_Emu.h"
#include "Gzip_Reader.h"

static const int fade_threshold = 10 * 1000;
static const int fade_length    = 8 * 1000;

static blargg_err_t log_err(blargg_err_t err)
{
    if (err)
        fprintf (stderr, "console: %s\n", err);
    return err;
}

static void log_warning(Music_Emu * emu)
{
    const char *str = emu->warning();
    if (str != NULL)
        fprintf (stderr, "console: %s\n", str);
}

/* Handles URL parsing, file opening and identification, and file
 * loading. Keeps file header around when loading rest of file to
 * avoid seeking and re-reading.
 */
class ConsoleFileHandler {
public:
    char *m_path;            // path without track number specification
    int m_track;             // track number (0 = first track)
    Music_Emu* m_emu;         // set to 0 to take ownership
    gme_type_t m_type;

    // Parses path and identifies file type
    ConsoleFileHandler(const char* path, VFSFile *fd = NULL);

    // Creates emulator and returns 0. If this wasn't a music file or
    // emulator couldn't be created, returns 1.
    int load(int sample_rate);

    // Deletes owned emu and closes file
    ~ConsoleFileHandler();

private:
    char m_header[4];
    Vfs_File_Reader vfs_in;
    Gzip_Reader gzip_in;
};

ConsoleFileHandler::ConsoleFileHandler(const char *path, VFSFile *fd)
{
    m_emu   = NULL;
    m_type  = 0;
    m_track = -1;

    const char * sub;
    uri_parse (path, NULL, NULL, & sub, & m_track);
    m_path = str_nget (path, sub - path);

    m_track -= 1;

    // open vfs
    if (fd != NULL)
        vfs_in.reset(fd);
    else if (log_err(vfs_in.open(m_path)))
        return;

    // now open gzip_reader on top of vfs
    if (log_err(gzip_in.open(&vfs_in)))
        return;

    // read and identify header
    if (!log_err(gzip_in.read(m_header, sizeof(m_header))))
    {
        m_type = gme_identify_extension(gme_identify_header(m_header));
        if (!m_type)
        {
            m_type = gme_identify_extension(m_path);
            if (m_type != gme_gym_type) // only trust file extension for headerless .gym files
                m_type = 0;
        }
    }
}

ConsoleFileHandler::~ConsoleFileHandler()
{
    gme_delete(m_emu);
    str_unref (m_path);
}

int ConsoleFileHandler::load(int sample_rate)
{
    if (!m_type)
        return 1;

    m_emu = gme_new_emu(m_type, sample_rate);
    if (m_emu == NULL)
    {
        log_err("Out of memory allocating emulator engine. Fatal error.");
        return 1;
    }

    // combine header with remaining file data
    Remaining_Reader reader(m_header, sizeof(m_header), &gzip_in);
    if (log_err(m_emu->load(reader)))
        return 1;

    // files can be closed now
    gzip_in.close();
    vfs_in.close();

    log_warning(m_emu);

#if 0
    // load .m3u from same directory( replace/add extension with ".m3u")
    char *m3u_path = g_strdup(m_path);
    char *ext = strrchr(m3u_path, '.');
    if (ext == NULL)
    {
        ext = g_strdup_printf("%s.m3u", m3u_path);
        g_free(m3u_path);
        m3u_path = ext;
    }

    Vfs_File_Reader m3u;
    if (!m3u.open(m3u_path))
    {
        if (log_err(m_emu->load_m3u(m3u))) // TODO: fail if m3u can't be loaded?
            log_warning(m_emu); // this will log line number of first problem in m3u
    }
#endif

    return 0;
}

static Tuple * get_track_ti(const char *path, const track_info_t *info, const int track)
{
    Tuple *ti = tuple_new_from_filename(path);

    if (ti != NULL)
    {
        tuple_set_str (ti, FIELD_ARTIST, info->author);
        tuple_set_str (ti, FIELD_ALBUM, info->game);
        tuple_set_str (ti, FIELD_TITLE, info->song);
        tuple_set_str (ti, FIELD_COPYRIGHT, info->copyright);
        tuple_set_str (ti, FIELD_CODEC, info->system);
        tuple_set_str (ti, FIELD_COMMENT, info->comment);

        if (track >= 0)
        {
            tuple_set_int(ti, FIELD_TRACK_NUMBER, track + 1);
            tuple_set_int(ti, FIELD_SUBSONG_ID, track + 1);
            tuple_set_int(ti, FIELD_SUBSONG_NUM, info->track_count);
        }
        else
            tuple_set_subtunes (ti, info->track_count, NULL);

        int length = info->length;
        if (length <= 0)
            length = info->intro_length + 2 * info->loop_length;
        if (length <= 0)
            length = audcfg.loop_length * 1000;
        else if (length >= fade_threshold)
            length += fade_length;
        tuple_set_int(ti, FIELD_LENGTH, length);
    }

    return ti;
}

extern "C" Tuple * console_probe_for_tuple(const char *filename, VFSFile *fd)
{
    ConsoleFileHandler fh(filename, fd);

    if (!fh.m_type)
        return NULL;

    if (!fh.load(gme_info_only))
    {
        track_info_t info;
        if (!log_err(fh.m_emu->track_info(&info, fh.m_track < 0 ? 0 : fh.m_track)))
            return get_track_ti(fh.m_path, &info, fh.m_track);
    }

    return NULL;
}

extern "C" bool_t console_play(const char *filename, VFSFile *file)
{
    int length, sample_rate;
    track_info_t info;

    // identify file
    ConsoleFileHandler fh(filename);
    if (!fh.m_type)
        return FALSE;

    if (fh.m_track < 0)
        fh.m_track = 0;

    // select sample rate
    sample_rate = 0;
    if (fh.m_type == gme_spc_type)
        sample_rate = 32000;
    if (audcfg.resample)
        sample_rate = audcfg.resample_rate;
    if (sample_rate == 0)
        sample_rate = 44100;

    // create emulator and load file
    if (fh.load(sample_rate))
        return FALSE;

    // stereo echo depth
    gme_set_stereo_depth(fh.m_emu, 1.0 / 100 * audcfg.echo);

    // set equalizer
    if (audcfg.treble || audcfg.bass)
    {
        Music_Emu::equalizer_t eq;

        // bass - logarithmic, 2 to 8194 Hz
        double bass = 1.0 - (audcfg.bass / 200.0 + 0.5);
        eq.bass = (long) (2.0 + pow( 2.0, bass * 13 ));

        // treble - -50 to 0 to +5 dB
        double treble = audcfg.treble / 100.0;
        eq.treble = treble * (treble < 0 ? 50.0 : 5.0);

        fh.m_emu->set_equalizer(eq);
    }

    // get info
    length = -1;
    if (!log_err(fh.m_emu->track_info(&info, fh.m_track)))
    {
        if (fh.m_type == gme_spc_type && audcfg.ignore_spc_length)
            info.length = -1;

        Tuple *ti = get_track_ti(fh.m_path, &info, fh.m_track);
        if (ti != NULL)
        {
            length = tuple_get_int(ti, FIELD_LENGTH);
            tuple_unref(ti);
            aud_input_set_bitrate(fh.m_emu->voice_count() * 1000);
        }
    }

    // start track
    if (log_err(fh.m_emu->start_track(fh.m_track)))
        return FALSE;

    log_warning(fh.m_emu);

    if (!aud_input_open_audio(FMT_S16_NE, sample_rate, 2))
        return FALSE;

    // set fade time
    if (length <= 0)
        length = audcfg.loop_length * 1000;
    if (length >= fade_threshold + fade_length)
        length -= fade_length / 2;
    fh.m_emu->set_fade(length, fade_length);

    while (!aud_input_check_stop())
    {
        /* Perform seek, if requested */
        int seek_value = aud_input_check_seek();
        if (seek_value >= 0)
            fh.m_emu->seek(seek_value);

        /* Fill and play buffer of audio */
        int const buf_size = 1024;
        Music_Emu::sample_t buf[buf_size];

        fh.m_emu->play(buf_size, buf);

        aud_input_write_audio(buf, sizeof(buf));

        if (fh.m_emu->track_ended())
            break;
    }

    return TRUE;
}
