// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef THIRD_PARTY_MOJO_SRC_MOJO_EDK_SYSTEM_UNIQUE_IDENTIFIER_H_
#define THIRD_PARTY_MOJO_SRC_MOJO_EDK_SYSTEM_UNIQUE_IDENTIFIER_H_

#include <stdint.h>
#include <string.h>

#include <string>

#include "base/containers/hash_tables.h"
#include "mojo/public/cpp/system/macros.h"
#include "third_party/mojo/src/mojo/edk/system/system_impl_export.h"

namespace mojo {
namespace system {

class UniqueIdentifier;

}  // namespace system
}  // namespace mojo

namespace BASE_HASH_NAMESPACE {

// Declare this before |UniqueIdentifier|, so that it can be friended.
template <>
struct hash<mojo::system::UniqueIdentifier>;

}  // BASE_HASH_NAMESPACE

namespace mojo {

namespace embedder {

class PlatformSupport;

}  // namespace embedder

namespace system {

// |UniqueIdentifier| is a POD class whose value is used to uniquely identify
// things.
class MOJO_SYSTEM_IMPL_EXPORT UniqueIdentifier {
 public:
  // This generates a new identifier. Uniqueness is "guaranteed" (i.e.,
  // probabilistically) for identifiers.
  static UniqueIdentifier Generate(embedder::PlatformSupport* platform_support);

  // This reconstitutes an identifier from its representation as a string
  // (generated by |ToString()|, below). On success, |*success| is set to true;
  // on failure, it is set to false (and the return value is undefined).
  static UniqueIdentifier FromString(const std::string& s, bool* success);

  // Yields a string representation of this identifier (which can be
  // reconstituted using |FromString()|. The result will consist of printable,
  // non-whitespace ASCII characters. It will *not* contain the following
  // characters: ", ', or \.
  std::string ToString() const;

  bool operator==(const UniqueIdentifier& other) const {
    return memcmp(data_, other.data_, sizeof(data_)) == 0;
  }
  bool operator!=(const UniqueIdentifier& other) const {
    return !operator==(other);
  }
  bool operator<(const UniqueIdentifier& other) const {
    return memcmp(data_, other.data_, sizeof(data_)) < 0;
  }

 private:
  friend BASE_HASH_NAMESPACE::hash<mojo::system::UniqueIdentifier>;

  explicit UniqueIdentifier() {}

  char data_[16];

  // Copying and assignment allowed.
};
static_assert(sizeof(UniqueIdentifier) == 16,
              "UniqueIdentifier has wrong size.");
// We want to be able to take any buffer and cast it to a |UniqueIdentifier|.
static_assert(MOJO_ALIGNOF(UniqueIdentifier) == 1,
              "UniqueIdentifier requires nontrivial alignment.");

}  // namespace system
}  // namespace mojo

namespace BASE_HASH_NAMESPACE {

template <>
struct hash<mojo::system::UniqueIdentifier> {
  size_t operator()(mojo::system::UniqueIdentifier unique_identifier) const {
    return base::HashInts64(
        *reinterpret_cast<uint64_t*>(unique_identifier.data_),
        *reinterpret_cast<uint64_t*>(unique_identifier.data_ +
                                     sizeof(uint64_t)));
  }
};

}  // BASE_HASH_NAMESPACE

#endif  // THIRD_PARTY_MOJO_SRC_MOJO_EDK_SYSTEM_UNIQUE_IDENTIFIER_H_
